#include "mbed.h"
#include "mDot.h"
#include <string>
#include <vector>
#include "MTSLog.h"

// TODO: use config file for these
// these options must match the settings on your Conduit
// uncomment the following lines and edit their values to match your configuration
static std::string config_network_name      = "Schakra_LoRa";
static std::string config_network_pass      = "Schakra_password";
static uint8_t config_frequency_sub_band    = 7;
const uint32_t cSleepPeriodSeconds          = 15;
const uint32_t cLoRaMaxPacketSize           = 246;
const float cSensorStartupSeconds           = 0.1f;
const float cAwakeLedCyclePeriod            = 0.3f;
const uint32_t cMaxJoinRetries              = 5;
const uint32_t cJoinRetryWait               = 5;

//#define DEVELOPER_BOARD

#ifdef DEVELOPER_BOARD
// spark fun moisture sensor
const PinName sensorPowerPin    = PA_11;    // D7           
const PinName analogInputPin    = PB_1;     // A0           

// Optional LED to demonstrate power to device/out of sleep. 
const PinName awakeLedPin       = PA_2;     // D1           

const int32_t onState = 0;
const int32_t offState = 1;
#else
// spark fun moisture sensor
const PinName sensorPowerPin    = PB_0;     // J3 pin 7     : active high
const PinName analogInputPin    = PB_1;     // J3 pin 8     : 0V-1V
                                            // J3 pin 10    : ground

// Optional LED to demonstrate power to device/out of sleep. 
const PinName awakeLedPin       = PA_5;     // J1 pin 15    : active low
                                            // J1 pin 19    : GND

const int32_t onState = 1;
const int32_t offState = 0;
#endif

DigitalOut  awakeLed    (awakeLedPin,offState);
DigitalOut  sensorPower (sensorPowerPin, offState);


void TurnOnSensor()
{
    sensorPower = onState;
    wait(cSensorStartupSeconds);
}

void TurnOffSensor()
{
    sensorPower = offState;
#ifdef DEVELOPER_BOARD
    wait(cSensorStartupSeconds);
#endif    
}
                                          
void awakeLedToggle(uint32_t cycles)
{
    for(uint32_t n = 0; n < cycles; n++)
    {
        awakeLed = onState; 
        wait(cAwakeLedCyclePeriod/10);
        awakeLed = offState; 
        wait(cAwakeLedCyclePeriod/10);
    }
}

int main()
{
    int32_t ret;
    
    AnalogIn    analogValue (analogInputPin);

    
    mDot* dot;
    std::vector<uint8_t> data;
    char buf [cLoRaMaxPacketSize];
    
    // indicate device out of sleep
    awakeLedToggle(1);
    
    // get a mDot handle
    dot = mDot::getInstance();
    
    // print library version information
    logInfo("version: %s", dot->getId().c_str());

    //*******************************************
    // configuration
    //*******************************************
    // reset to default config so we know what state we're in
    dot->resetConfig();
    
    dot->setLogLevel(mts::MTSLog::TRACE_LEVEL);

    // max output power    
    dot->setTxPower (20);
    
    // disable Ack return
    dot->setAck(0);

    // set up the mDot with our network information: frequency sub band, network name, and network password
    logInfo("setting frequency sub band");
    if ((ret = dot->setFrequencySubBand(config_frequency_sub_band)) != mDot::MDOT_OK)
    {
        logError("failed to set frequency sub band %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    logInfo("setting network name");
    if ((ret = dot->setNetworkName(config_network_name)) != mDot::MDOT_OK)
    {
        logError("failed to set network name %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    logInfo("setting network password");
    if ((ret = dot->setNetworkPassphrase(config_network_pass)) != mDot::MDOT_OK)
    {
        logError("failed to set network password %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    logInfo("saving config");
    if (! dot->saveConfig())
    {
        logError("failed to save configuration");
    }    

    //*******************************************
    // end of configuration
    //*******************************************

    // join the network
    logInfo("joining network");
    for(int32_t retry = 0; retry <= cMaxJoinRetries; retry++)
    {
        if((ret = dot->joinNetwork()) != mDot::MDOT_OK)
        {
            if(retry >= cMaxJoinRetries)
            {
                dot->sleep(cSleepPeriodSeconds);
            }
            else
            {
                logError("failed to join network %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
                wait(cJoinRetryWait);
            }
        }
        else
        {
            logInfo("joined network!!!");
            break;
        }
    }

    awakeLedToggle(2);

    TurnOnSensor();
    wait(cSensorStartupSeconds);
    float reading = analogValue.read();
    TurnOffSensor();

    // format data for sending to the gateway
    vector<uint8_t> buffer;
    char* p = buf;
    sprintf( buf, "Moisture=%f volts", reading);
    while( *p != 0 )
    {
        buffer.push_back( *p);
        p++;
    }

    // send the data
    // ACKs are enabled by default, so we're expecting to get one back
    if ((ret = dot->send(buffer)) != mDot::MDOT_OK)
    {
        logError("failed to send", ret, mDot::getReturnCodeString(ret).c_str());
    }
    else
    {
        logInfo("successfully sent data to gateway");
    }

    awakeLedToggle(3);

    // Deep sleep and awake to RTC. Wake is essentially reboot.
    dot->sleep(cSleepPeriodSeconds);

    return 0;
}
