/*
 * Copyright (c) 2017, CATIE, All Rights Reserved
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "mbed.h"
//#include "EthernetInterface.h"

#include "aws_iot_config.h"
#include "aws_iot_mqtt_client_interface.h"

namespace {
#define READ_DURATION 500
#define DEMO_PAYLOAD "Hello, world!"
#define PUBLISH_TOPIC "sdkTest/sub"
#define PUBLISH_TOPIC_LEN 11
}

void subscribe_callback(AWS_IoT_Client *client, char *topic_name,
		uint16_t topic_name_length, IoT_Publish_Message_Params *params,
		void *data)
{
    printf("Subscribe callback");
    printf("%.*s\t%.*s", topic_name_length, topic_name, (int)params->payloadLen, (char *)params->payload);
}

void disconnect_handler(AWS_IoT_Client *client, void *data)
{
    printf("MQTT Disconnect");
    IoT_Error_t rc = IOT_FAILURE;

    if (!client) {
        return;
    }

    if (aws_iot_is_autoreconnect_enabled(client)) {
        printf("Auto Reconnect is enabled, Reconnecting attempt will start now");
    } else {
        printf("Auto Reconnect not enabled. Starting manual reconnect...");
        rc = aws_iot_mqtt_attempt_reconnect(client);
        if (NETWORK_RECONNECTED == rc) {
            printf("Manual Reconnect Successful");
        } else {
            printf("Manual Reconnect Failed - %d", rc);
        }
    }
}

//EthernetInterface net;

int main()
{
    //net.connect();    
    AWS_IoT_Client client;

    /*
     * MQTT Init
     */

    IoT_Client_Init_Params mqtt_init_params = iotClientInitParamsDefault;
    mqtt_init_params.enableAutoReconnect = false; // We enable this later below
    char mqtt_host[] = AWS_IOT_MQTT_HOST;
    mqtt_init_params.pHostURL = mqtt_host;
    mqtt_init_params.port = AWS_IOT_MQTT_PORT;
    mqtt_init_params.disconnectHandler = disconnect_handler;
    mqtt_init_params.disconnectHandlerData = NULL;

    IoT_Error_t mqtt_init_status = aws_iot_mqtt_init(&client, &mqtt_init_params);
    if (mqtt_init_status != IOT_SUCCESS) {
        printf("aws_iot_mqtt_init returned error : %d ", mqtt_init_status);
        while (true) { Thread::wait(1000); }
    }

    /*
     * MQTT Connect
     */

    IoT_Client_Connect_Params connect_params = iotClientConnectParamsDefault;

    connect_params.keepAliveIntervalInSec = 10;
    connect_params.isCleanSession = true;
    connect_params.MQTTVersion = MQTT_3_1_1;
    char client_id[] = AWS_IOT_MQTT_CLIENT_ID;
    connect_params.pClientID = client_id;
    connect_params.clientIDLen = (uint16_t)strlen(AWS_IOT_MQTT_CLIENT_ID);
    connect_params.isWillMsgPresent = false;

    IoT_Error_t mqtt_connect_status = aws_iot_mqtt_connect(&client, &connect_params);
    if (mqtt_connect_status != IOT_SUCCESS) {
        printf("Error(%d) connecting to %s:%d", mqtt_connect_status,
                mqtt_init_params.pHostURL, mqtt_init_params.port);
        while (true) { Thread::wait(1000); }
    }

    /*
     * Set MQTT autoreconnect
     */

    IoT_Error_t mqtt_autoreconnect_status = aws_iot_mqtt_autoreconnect_set_status(&client, true);
    if (mqtt_autoreconnect_status != IOT_SUCCESS) {
        printf("Unable to set Auto Reconnect to true - %d", mqtt_autoreconnect_status);
        while (true) { Thread::wait(1000); }
    }

    /*
     * MQTT Subscribe
     */

    IoT_Error_t mqtt_subscribe_status = aws_iot_mqtt_subscribe(
		    &client, "sdkTest/sub", 11, QOS0, subscribe_callback, NULL);
    if (mqtt_subscribe_status != IOT_SUCCESS) {
        printf("Error subscribing: %d ", mqtt_subscribe_status);
        while (true) { Thread::wait(1000); }
    }

    /*
     * MQTT Publish
     */

    IoT_Publish_Message_Params mqtt_publish_params;
    mqtt_publish_params.qos = QOS0;
    char payload[] = DEMO_PAYLOAD;
    mqtt_publish_params.payload = payload;
    mqtt_publish_params.payloadLen = strlen(payload);
    mqtt_publish_params.isRetained = 0;

    // Main loop
    IoT_Error_t mqtt_status;
    while (true) {
        //Max time the yield function will wait for read messages
        mqtt_status = aws_iot_mqtt_yield(&client, READ_DURATION);
        if (mqtt_status == NETWORK_ATTEMPTING_RECONNECT) {
            // If the client is attempting to reconnect we will skip the rest of the loop.
            continue;
        }

        mqtt_status = aws_iot_mqtt_publish(
                &client, PUBLISH_TOPIC, PUBLISH_TOPIC_LEN, &mqtt_publish_params);
        if (mqtt_status != IOT_SUCCESS) {
            printf("Error publishing: %d", mqtt_status);
        }
    }
}
