/** 
 * PushDuration Library by Jens Strümper. For questions or suggestions on how to improve the 
 * library please contact me via email: jens.struemper@gmail.com
**/
#ifndef __PUSHDURATION_H_INCLUDED__
#define __PUSHDURATION_H_INCLUDED__

namespace js{

/**
 * PushDuration calls a function based on the duration of a button push
 *
 * The following code demonstrates the usage of the PushDuration library. 
 * The library is meant for cases where different callbank functions are triggered by 
 * different button push durations. E.g function "state1" is called when the button is
 * released after 1 second, "state4" is called when the button is released after 4 seconds
 * and so on. Please note that the implementation is far from being perfect. E.g. it 
 * will not work if you instantiate the class in ram. 

* Paramters are:-
 *  <ul>
 *  <li> Key value table for function <-> duration assignment. </li>
 *  <li> PinName used for Button </li> 
 *  <li> Interval - typically "1" for full seconds </li>
 *  </ul>
 *
 *
 * Example:
 * @code


 * #include "mbed.h"
 * #include "PushDuration.h"

 * void state1() {
 *    //function called after 1 seconds
 *    printf("state1\n");
 * }
 * void state4() {
 *    //function called after 4 seconds
 *   printf("state4\n");
 * }
 * void state10() {
 *   //function called after 4 seconds
 *   printf("state8\n");
 * }

 * PinName pin(p17);

 * //Action Table specifiying push duration and callback function
 * const action action_table[] =
 * {
 *     { 1, state1 },
 *     { 4, state4 },
 *     { 10, state10 },
 * };

 * int main()
 * //creating the ButtonHandler Object by specifiying arraysize, action table, pin and interval in seconds 
 * {
 *     ButtonHandler green(
 *     sizeof(action_table)/sizeof(action_table[0]),
 *     action_table,
 *     pin,
 *     2
 *     );

 * //enabling Button Handler
 *     green.enable();
 *     while(1){
 *         printf("alive\n");
 *         wait(5);
 *      }
 * }
**/

/**
 * struct to define duration and 
 * callback function
**/

struct action
{
    int counter_limit;
    void (*transition)(void);
};

/**
 * struct to hold table size and p
 * pointer to the first element of the table
**/

struct action_list
{
    size_t size;
    const action *table;
};


class ButtonHandler {
public:
    /**
     * ButtonHandler Constructor:
     *
     * @param table_size sitze_t
     * @param begin  const action*
     * @param pin PinName
     * @param seconds float
    **/
    ButtonHandler(std::size_t table_size, const action* begin, PinName pin, float seconds ): buttonPin(pin){
        mTable.size = table_size;
        mTable.table = begin;
        intervalInSeconds = seconds;
        printf("initialized\n");
    }

    /**
     * Enables Button Handler 
    **/
    void enable();
    
    /**
     * Disables Button Handler 
    **/
    
    void disable();
    
    /**
     * ButtonHandler Destructor 
    **/
    virtual ~ButtonHandler() {
        disable();
    }
protected:
    void press();
    void secondsCount();
    void release();
    void react(int counter) const;
    
private:
    volatile unsigned counter;
    InterruptIn buttonPin;
    float intervalInSeconds;
    action_list mTable;
    Ticker ticker;
    
};
}; //name space ends

using namespace js; 

#endif //__PUSHDURATION_H_INCLUDED__
