#include"SI114x.h"

/// COMMAND LIST
    const uint8_t SI114x::CMD_PARAM_QUERY           = 0x80; // Reads the parameter pointed to by bitfield [4:] and writes value to PARAM_RD
    const uint8_t SI114x::CMD_PARAM_SET             = 0xA0; // Sets the parameter pointed to by bitfield [4:] with value in PARAM_WR and writes value out to PARAM_RD
    const uint8_t SI114x::CMD_NOP                   = 0x00; // Forces a zero into the response register
    const uint8_t SI114x::CMD_RESET                 = 0x01; // Performs a software reset of the firmware
    const uint8_t SI114x::CMD_BUSADDR               = 0x02; // Modifies the I2C address
    // Reserved: 0x03, 0x04
    const uint8_t SI114x::CMD_PS_FORCE              = 0x05; // Forces a single PS measurement
    const uint8_t SI114x::CMD_ALS_FORCE             = 0x06; // Forces a single ALS measurement
    const uint8_t SI114x::CMD_PSALS_FORCE           = 0x07; // Forces a single PS and ALS measurement
    // Reserved: 0x08
    const uint8_t SI114x::CMD_PS_PAUSE              = 0x09; // Pauses autonomous PS
    const uint8_t SI114x::CMD_ALS_PAUSE             = 0x0A; // Pauses autonomous ALS
    const uint8_t SI114x::CMD_PSALS_PAUSE           = 0x0B; // Pauses autonomous PS and ALS
    // Reserved: 0x0C
    const uint8_t SI114x::CMD_PS_AUTO               = 0x0D; // Starts/restarts an autonomous PS loop
    const uint8_t SI114x::CMD_ALS_AUTO              = 0x0E; // Starts/restarts an autonomous ALS loop
    const uint8_t SI114x::CMD_PSALS_AUTO            = 0x0F; // Starts/restarts an autonomous PS and ALS loop
    // Reserved: all 0x1* except:
    const uint8_t SI114x::CMD_GET_CAL_INDEX         = 0x11; // Reports calibration index 
    const uint8_t SI114x::CMD_GET_CAL               = 0x12; // Reports calibration data to I2C registers 0x22-0x2D

/// PARAMETERS
    const uint8_t SI114x::PARAM_I2C_ADDR            = 0x00;
    const uint8_t SI114x::PARAM_CHLIST              = 0x01;
    const uint8_t SI114x::PARAM_PSLED12_SELECT      = 0x02;
    const uint8_t SI114x::PARAM_PSLED3_SELECT       = 0x03;
    const uint8_t SI114x::PARAM_FILTER_EN           = 0x04;
    const uint8_t SI114x::PARAM_PS_ENCODING         = 0x05;
    const uint8_t SI114x::PARAM_ALS_ENCODING        = 0x06;
    const uint8_t SI114x::PARAM_PS1_ADC_MUX         = 0x07;
    const uint8_t SI114x::PARAM_PS2_ADC_MUX         = 0x08;
    const uint8_t SI114x::PARAM_PS3_ADC_MUX         = 0x09;
    const uint8_t SI114x::PARAM_PS_ADC_COUNTER      = 0x0A;
    const uint8_t SI114x::PARAM_PS_ADC_CLKDIV       = 0x0B;
    const uint8_t SI114x::PARAM_PS_ADC_GAIN         = 0x0B;
    const uint8_t SI114x::PARAM_PS_ADC_MISC         = 0x0C;
    const uint8_t SI114x::PARAM_VIS_ADC_MUX         = 0x0D;
    const uint8_t SI114x::PARAM_IR_ADC_MUX          = 0x0E;
    const uint8_t SI114x::PARAM_AUX_ADC_MUX         = 0x0F;
    const uint8_t SI114x::PARAM_ALSVIS_ADC_COUNTER  = 0x10;
    const uint8_t SI114x::PARAM_ALSVIS_ADC_CLKDIV   = 0x11;
    const uint8_t SI114x::PARAM_ALSVIS_ADC_GAIN     = 0x11;
    const uint8_t SI114x::PARAM_ALSVIS_ADC_MISC     = 0x12;
    const uint8_t SI114x::PARAM_ALS_HYST            = 0x16;
    const uint8_t SI114x::PARAM_PS_HYST             = 0x17;
    const uint8_t SI114x::PARAM_PS_HISTORY          = 0x18;
    const uint8_t SI114x::PARAM_ALS_HISTORY         = 0x19;
    const uint8_t SI114x::PARAM_ADC_OFFSET          = 0x1A;
    const uint8_t SI114x::PARAM_SLEEP_CTRL          = 0x1B;
    const uint8_t SI114x::PARAM_LED_RECOVERY        = 0x1C;
    const uint8_t SI114x::PARAM_ALSIR_ADC_COUNTER   = 0x1D;
    const uint8_t SI114x::PARAM_ALSIR_ADC_CLKDIV    = 0x1E;
    const uint8_t SI114x::PARAM_ALSIR_ADC_GAIN      = 0x1E;
    const uint8_t SI114x::PARAM_ALSIR_ADC_MISC      = 0x1F;

/// RESPONSE REGISTER
    const uint8_t SI114x::RES_NO_ERROR              = 0x00; // NO_ERROR: 0000:cccc
    const uint8_t SI114x::RES_INVALID_SETTING       = 0x80; // Clear using NOP command
    const uint8_t SI114x::RES_PS1_ADC_OVERFLOW      = 0x88; // Indicates proximity channel 1 conversion overflow
    const uint8_t SI114x::RES_PS2_ADC_OVERFLOW      = 0x89; // Indicates proximity channel 2 conversion overflow
    const uint8_t SI114x::RES_PS3_ADC_OVERFLOW      = 0x8A; // Indicates proximity channel 3 conversion overflow
    const uint8_t SI114x::RES_ALS_VIS_ADC_OVERFLOW  = 0x8C; // Indicates visible light ambient light channel conversion overflow
    const uint8_t SI114x::RES_ALS_IR_ADC_OVERFLOW   = 0x8D; // Indicates infrared ambient light channel conversion overflow
    const uint8_t SI114x::RES_AUX_ADC_OVERFLOW      = 0x8E; // Indicates auxiliary channel conversion overflow

/// I2C Registers
    const uint8_t SI114x::REG_PART_ID               = 0x00;
    const uint8_t SI114x::REG_REV_ID                = 0x01;
    const uint8_t SI114x::REG_SEQ_ID                = 0x02;
    const uint8_t SI114x::REG_INT_CFG               = 0x03; // [0] INT_OE
    const uint8_t SI114x::REG_IRQ_ENABLE            = 0x04; // [4] PS3_IE [3] PS2_IE [2] PS1_IE [0] ALS_IE
    const uint8_t SI114x::REG_IRQ_MODE1             = 0x05; // 
    const uint8_t SI114x::REG_IRQ_MODE2             = 0x06; // 
    const uint8_t SI114x::REG_HW_KEY                = 0x07;
    const uint8_t SI114x::REG_MEAS_RATE             = 0x08;
    const uint8_t SI114x::REG_ALS_RATE              = 0x09;
    const uint8_t SI114x::REG_PS_RATE               = 0x0A; // 
    const uint8_t SI114x::REG_ALS_LO_TH_LSB         = 0x0B; // 
    const uint8_t SI114x::REG_ALS_LO_TH_MSB         = 0x0C; // 
    const uint8_t SI114x::REG_ALS_HI_TH_LSB         = 0x0D; // 
    const uint8_t SI114x::REG_ALS_HI_TH_MSB         = 0x0E; // 
    const uint8_t SI114x::REG_PS_LED21              = 0x0F; // [7:4] LED2_I [3:0] LED1_I
    const uint8_t SI114x::REG_PS_LED3               = 0x10; // [3:0] LED3_I
    const uint8_t SI114x::REG_PS1_TH_LSB            = 0x11;
    const uint8_t SI114x::REG_PS1_TH_MSB            = 0x12;
    const uint8_t SI114x::REG_PS2_TH_LSB            = 0x13;
    const uint8_t SI114x::REG_PS2_TH_MSB            = 0x14;
    const uint8_t SI114x::REG_PS3_TH_LSB            = 0x15;
    const uint8_t SI114x::REG_PS3_TH_MSB            = 0x16;
    const uint8_t SI114x::REG_PARAM_WR              = 0x17;
    const uint8_t SI114x::REG_COMMAND               = 0x18;
    // Undefined: 0x19..0x1F
    const uint8_t SI114x::REG_RESPONSE              = 0x20;
    const uint8_t SI114x::REG_IRQ_STATUS            = 0x21; // [5] CMD_INT [4] PS3_INT [3] PS2_INT [2] PS1_INT [1:0] ALS_INT
    const uint8_t SI114x::REG_ALS_VIS_DATA0         = 0x22;
    const uint8_t SI114x::REG_ALS_VIS_DATA1         = 0x23;
    const uint8_t SI114x::REG_ALS_IR_DATA0          = 0x24;
    const uint8_t SI114x::REG_ALS_IR_DATA1          = 0x25;
    const uint8_t SI114x::REG_PS1_DATA0             = 0x26;
    const uint8_t SI114x::REG_PS1_DATA1             = 0x27;
    const uint8_t SI114x::REG_PS2_DATA0             = 0x28;
    const uint8_t SI114x::REG_PS2_DATA1             = 0x29;
    const uint8_t SI114x::REG_PS3_DATA0             = 0x2A;
    const uint8_t SI114x::REG_PS3_DATA1             = 0x2B;
    const uint8_t SI114x::REG_AUX_DATA0             = 0x2C;
    const uint8_t SI114x::REG_AUX_DATA1             = 0x2D;
    const uint8_t SI114x::REG_PARAM_RD              = 0x2E;
    // Undefined: 0x2F
    const uint8_t SI114x::REG_CHIP_STAT             = 0x30;
    // Aliases
    const uint8_t SI114x::REG_UCOEF0                = 0x13;
    const uint8_t SI114x::REG_UCOEF1                = 0x14;
    const uint8_t SI114x::REG_UCOEF2                = 0x15;
    const uint8_t SI114x::REG_UCOEF3                = 0x16;
    const uint8_t SI114x::REG_MEAS_RATE_LSB         = 0x08;
    const uint8_t SI114x::REG_MEAS_RATE_MSB         = 0x09;
    
    
/// VALUES
    const uint8_t SI114x::VAL_HW_KEY                = 0x17; // System must write 0x17 to REG_HW_KEY for proper operation
    const uint8_t SI114x::VAL_UCOEF0                = 0x7B;
    const uint8_t SI114x::VAL_UCOEF1                = 0x6B;
    const uint8_t SI114x::VAL_UCOEF2                = 0x01;
    const uint8_t SI114x::VAL_UCOEF3                = 0x00;
    const uint8_t SI114x::VAL_INTOE                 = 0x01;
    const uint8_t SI114x::VAL_INTMODE               = 0x02;
    const uint8_t SI114x::VAL_EN_ALS_VIS            = 0x10;
    const uint8_t SI114x::VAL_EN_ALS_IR             = 0x20;
    const uint8_t SI114x::VAL_EN_AUX                = 0x40;
    const uint8_t SI114x::VAL_EN_UV                 = 0x80;    
 
float SI114x::m_norm_tab[2][8];
 
extern Serial ser; 
   
SI114x::SI114x(void) : m_pI2C(NULL),m_address(0x00),m_bOk(false),m_name(std::string("Silicon Labs SI114x")),
    m_part_id(0x00),m_vis_norm(0.0f),m_ir_norm(0.0f),m_uv_norm(0.01f),m_mode(MODE_SLEEP) 
{
}
 
SI114x::~SI114x(void) {
    m_pI2C = NULL;
    m_address = 0x00;
    m_bOk = false;
}

bool SI114x::init(I2C& i2c, uint8_t address) {
    m_bOk  = false;
    m_pI2C = &i2c;
    m_address = address<<1;
    if (!reset()) return false;
    m_part_id = part_id();
    switch(m_part_id) {
        case 0x45: m_name = std::string("Silicon Labs SI1145"); break;
        case 0x46: m_name = std::string("Silicon Labs SI1146"); break;
        case 0x47: m_name = std::string("Silicon Labs SI1147"); break;
        default  : m_name = std::string("SI114x - Unknown"); 
    }
    if (!read_calibration()) return false;
    
    for (int g=0; g<8; g++) {
        m_norm_tab[0][g] = 2.0f/pow(2.0f,float(g));         // low range 
        m_norm_tab[1][g] = m_norm_tab[0][g]*14.5f;          // high range
    }
    
    m_bOk = true;
    return m_bOk;
}

extern Serial ser;

bool SI114x::start( const illum_t& ill, 
                    const range_t& visrange, const gain_t& visgain, 
                    const range_t& irrange,  const gain_t& irgain,
                    const mode_t& mode, float meas_delay, DigitalIn IRQ_PIN) {
    if (m_pI2C==NULL) return false;
    uint8_t meas_rate = 0;
    uint8_t als_rate = 0;
    uint8_t ps_rate = 0;
    if (mode==MODE_AUTO) {
        ser.printf("meas_delay %.1f\r\n",meas_delay);
        uint16_t wake_delay = uint16_t(min(meas_delay*32000.0f,65535.0f));
        uint16_t n_wakes = max(uint16_t(meas_delay/(float(wake_delay)/32000.0f)+0.5f),uint16_t(1));
        ser.printf("wake = %i\r\nN = %i\r\n",wake_delay,n_wakes);
        meas_rate = compress(wake_delay);
        als_rate = compress(n_wakes);
        ser.printf("MEAS = %i\r\nALS = %i\r\n",uncompress(meas_rate),uncompress(als_rate));
        ser.printf("RAW: MEAS=0x%.2x ALS=0x%.2x\r\n",meas_rate,als_rate);
    }
    
    m_illum = ill;
    
    m_visrange = visrange;
    m_visgain  = visgain;
    set_vis_param();
    
    m_ir_norm  = m_norm_tab[uint8_t(irrange)][uint8_t(irgain)];
    m_uv_norm  = m_vis_norm/200.0f;

    if (!write8(REG_MEAS_RATE,meas_rate)) return false;
    if (!write8(REG_ALS_RATE,als_rate)) return false;
    if (!write8(REG_PS_RATE,ps_rate)) return false;

    // Measure using small photodiode (less noise, lower sensitivity)
    if (!write_param(PARAM_IR_ADC_MUX,0x00)) return false;
        
    
    // report 16 msb of the measurement
    if (!write_param(PARAM_ALS_ENCODING,0)) return false;
    
    if (m_mode==MODE_AUTO) als_auto();
    
    m_mode = mode;
    
    // TODO
    return true;
}

bool SI114x::stop(void) {
    if (m_pI2C==NULL) return false;
    // TODO
    return true;
}

bool SI114x::done(void) {
    if (m_pI2C==NULL) return false;
    // TODO
    return true;
}

const bool& SI114x::is_ok(void) const {
    return m_bOk;
}

const std::string& SI114x::name(void) const {
    return m_name;
}

bool SI114x::get(uint16_t& uv_index, uint16_t& visible, uint16_t& ir, bool update) {
    if (m_pI2C==NULL) return false;
    uint8_t res = 0x00;    
    uv_index = visible = ir = 0;
    switch(m_mode) {
        case MODE_FORCED:
            // Force conversion of visible light, uv index, ir light
            if (!write8(REG_IRQ_STATUS, 0x01)) return false;
            if (!write_param(PARAM_CHLIST, VAL_EN_ALS_VIS | VAL_EN_ALS_IR | VAL_EN_UV)) return false;
            if (!command(CMD_ALS_FORCE)) return false;
            wait_ms(250);
            do {
                if (!read8(res,REG_IRQ_STATUS)) return false;
                if ((res&0x1)!=0x1) wait_ms(5);
            } while(res&0x01!=0x01);
            // fall through            
        case MODE_AUTO:
            if (!read16(visible, REG_ALS_VIS_DATA0)) return false;
            if (!read16(uv_index,REG_AUX_DATA0)) return false;
            if (!read16(ir,      REG_ALS_IR_DATA0)) return false;
            visible  = max(uint16_t(256),visible)-256;
            ir       = max(uint16_t(256),ir)-256;
            // auto-adjust gain
            if (update) {
                if (!update_ranges(visible,ir)) return false;
            }
            break;
        case MODE_SLEEP:
            return false; 
    }
    return true;
}

bool SI114x::get(float& uv_index, float& visible, float& ir) {
    if (m_pI2C==NULL) return false;
    uint16_t _uv,_vis,_ir;
    if (!get(_uv,_vis,_ir,false)) return false;
    uv_index = float(_uv)*m_uv_norm;
    visible  = float(_vis)*m_vis_norm;
    ir       = float(_ir)*m_ir_norm;
    if (!update_ranges(_vis,_ir)) return false;
    return true;
}

bool SI114x::update_ranges(uint16_t vis, uint16_t ir) {
    // inverse sensitivity order
    // GAIN   RANGE_LOW  RANGE_HIGH
    //  0     [ 2  ]  4  [ 29 ]   0
    //  1     [ 1  ]  6  [14.5]   1
    //  2     [1/2 ]  8  [7.25]   2
    //  3     [1/4 ] 10  [3.63]   3
    //  4     [1/8 ] 12  [1.81]   5
    //  5     [1/16] 13  [0.91]   7
    //  6     [1/32] 14  [0.45]   9
    //  7     [1/64] 15  [0.23]  11
    static const uint8_t code_tab[2][8] = {
        {4,6,8,10,12,13,14,15}, // low range
        {0,1,2, 3, 5, 7, 9,11}  // high range
    };
    static const uint8_t rev_code_tab[16][2] = {
        {1,0}, // range, gain
        {1,1}, {1,2}, {1,3}, {0,0}, {1,4}, {0,1}, {1,5}, {0,2}, {1,6}, {0,3}, {1,7}, {0,4}, {0,5}, {0,6}, {0,7}
    };
    bool bOverflow = false;
    uint8_t code = code_tab[uint8_t(m_visrange)][uint8_t(m_visgain)];
    // tries to bring vis adc values to the range 4096..16384
    
    // adjust code by increasing for large values and decreasing for small values
    if (vis>=32768) { // overflow, decrease by 3 steps
        bOverflow = true;
        code = code>3 ? code-3 : 0;
        goto visnext;
    }
    if (vis>=16384) { // decrease by 1 step, target ~8192
        code = code>1 ? code-1 : 0;
        goto visnext;
    }
    if (vis<2048) { // increase by 2 steps, target ~4096
        code = min(15,code+2);
        goto visnext;
    }
    if (vis<4096) { // increase by 1 step, target ~8192
        code = min(15,code+1);
        goto visnext;
    }
    goto visdone; // no change
visnext:
    m_visrange = range_t(rev_code_tab[code][0]);
    m_visgain  = gain_t(rev_code_tab[code][1]);
    //ser.printf("VIS param: %s, gain=%i norm=%f\r\n",m_visrange==RANGE_HIGH ? "Hi" : "Lo",uint8_t(m_visgain),m_vis_norm);
    if (!set_vis_param()) return false;
    m_uv_norm = m_vis_norm*0.005f;
visdone:

    code = code_tab[uint8_t(m_irrange)][uint8_t(m_irgain)];
    if (ir>=32768) { // overflow, decrease by 2 steps
        bOverflow = true;
        code = code>3 ? code-3 : 0;
        goto irnext;
    }
    if (ir>=16384) { // decrease by 1 step
        code = code>1 ? code-1 : 0;
        goto irnext;
    }
    if (ir<2048) { // increase by 2 steps
        code = min(15,code+2);
        goto irnext;
    }
    if (ir<4096) { // increase by 1 step
        code = min(15,code+1);
        goto irnext;
    }
    goto irdone; // no change
irnext:
    m_irrange = range_t(rev_code_tab[code][0]);
    m_irgain  = gain_t(rev_code_tab[code][1]);
    //ser.printf("IR param: %s, gain=%i norm=%f\r\n",m_irrange==RANGE_HIGH ? "Hi" : "Lo",uint8_t(m_irgain),m_ir_norm);
    if (!set_ir_param()) return false;
irdone:
    return !bOverflow;
}
        
bool SI114x::set_vis_param(void) {
    m_vis_norm = m_norm_tab[uint8_t(m_visrange)][uint8_t(m_visgain)];
    switch(m_illum) {
        case ILLUM_SUNLIGHT:        m_vis_norm /= 0.282f;   break;
        case ILLUM_INCANDESCENT:    m_vis_norm /= 0.319f;   break;
        case ILLUM_FLUORESCENT:     m_vis_norm /= 0.146f;   break;
    }
    // Set ADC recovers
    if (m_visrange==RANGE_HIGH) {
        if (!write_param(PARAM_ALSVIS_ADC_COUNTER,0)) return false;
        // TODO: ALS_VIS_ADC_CLKDIV
    } else {
        if (!write_param(PARAM_ALSVIS_ADC_COUNTER,7-uint8_t(m_visgain))) return false;
    }
    if (!write_param(PARAM_ALSVIS_ADC_GAIN,uint8_t(m_visgain))) return false;
    if (!write_param(PARAM_ALSVIS_ADC_MISC,uint8_t(m_visrange)<<5)) return false;
    return true;
}

bool SI114x::set_ir_param(void) {
    m_ir_norm = m_norm_tab[uint8_t(m_irrange)][uint8_t(m_irgain)];
    switch(m_illum) {
        case ILLUM_SUNLIGHT:        m_ir_norm  /= 2.44f;    break;
        case ILLUM_INCANDESCENT:    m_ir_norm  /= 8.46f;    break;
        case ILLUM_FLUORESCENT:     m_ir_norm  /= 0.71f;    break;            
    }
    if (m_irrange==RANGE_HIGH) {
        if (!write_param(PARAM_ALSIR_ADC_COUNTER,0)) return false;
    } else {
        if (!write_param(PARAM_ALSIR_ADC_COUNTER,7-uint8_t(m_irgain))) return false;
    }
    if (!write_param(PARAM_ALSIR_ADC_GAIN,uint8_t(m_irgain))) return false;
    if (!write_param(PARAM_ALSIR_ADC_MISC,uint8_t(m_irrange)<<5)) return false;
    return true;
}
    
    

// ============================== LOW LEVEL API ==============================
bool SI114x::setInterrupt(bool enable) {
    return write8(REG_INT_CFG,(enable? VAL_INTOE : 0));
}

bool SI114x::getInterrupt(bool& enable) {
    uint8_t res = 0x00;
    if (!read8(res,REG_INT_CFG)) return false;
    enable = (res&VAL_INTOE)!=0;
    return true;
}


bool SI114x::wait_until_sleep(void) {
    if (m_pI2C==NULL) return false;
    uint8_t res;
    while(1) {
        if (!read8(res,REG_CHIP_STAT)) return false;
        if (res==1) break;
        else wait_ms(1);
    }
    return true;
}

bool SI114x::reset(void) {    
    // 1. Reset measurement rate
    if (!write8(REG_MEAS_RATE,0x00))        return false;
    
    // 2. Pause all
    if (!pause_all())                       return false;
    
    // 3. Clearing of registers
    if (!write8(REG_MEAS_RATE,0x00))        return false;
    if (!write8(REG_IRQ_ENABLE,0x00))       return false;
    if (!write8(REG_IRQ_MODE1,0x00))        return false;
    if (!write8(REG_IRQ_MODE2,0x00))        return false;
    if (!write8(REG_INT_CFG,0x00))          return false;
    if (!write8(REG_IRQ_STATUS,0xFF))       return false;
    
    // 4. reset command
        // a. reset response register
        if (!write8(REG_COMMAND,CMD_NOP))   return false;
        // b. verify response register == 0x00
        uint8_t res = 0x00;
        if (!read8(res,REG_RESPONSE))       return false;
        if (res!=0x00)                      return false;
        // c. write command
        if (!write8(REG_COMMAND,CMD_RESET)) return false;
    
    // 5. delay for 10 ms
    wait_ms(10);
    
    // 6. write hardware key
    if (!write8(REG_HW_KEY,VAL_HW_KEY))     return false;
    
    // 7. write default UCOEFS
    if (!write8(REG_UCOEF0,0x29)) return false;
    if (!write8(REG_UCOEF1,0x89)) return false;
    if (!write8(REG_UCOEF2,0x02)) return false;
    if (!write8(REG_UCOEF3,0x00)) return false;
    return true;
}

bool SI114x::command(uint8_t cmd, uint8_t* response) {
    if (response) *response=0xFF;
    if (m_pI2C==NULL) return false;
    
    uint8_t res = 0x00;
    uint8_t retval = 0x00;
    
    // 1. Get the response register contents
    if (!read8(res,REG_RESPONSE)) return false;
    
    // 2. Make sure response register is consistent
    while(1) {
        if (!wait_until_sleep()) return false;
        if (cmd==CMD_NOP) break;
        if (!read8(retval,REG_RESPONSE)) return false;
        if (retval==res) break;
        else res = retval;
    }
    
    // 3. send the command
    if (!write8(REG_COMMAND,cmd)) return false;  
    
    // 4. expect a change in response
    while (1) {
        if (cmd==CMD_NOP) break;
        if (!read8(retval,REG_RESPONSE)) return false;
        if (retval!=res) break;
    }
    
    if (response) *response = retval;
    return true;
}

bool SI114x::x_command(uint8_t cmd, uint8_t* response) {
    if (m_pI2C==NULL) return false;
    uint8_t old_value = 0x00;
    if (!read8(old_value,REG_RESPONSE)) return false;
    if (!write8(REG_COMMAND,cmd)) return false;
    if (cmd==CMD_RESET) return true;
    
    char i = 0;
    uint8_t res = 0x00;
    do {
        if (!read8(res,REG_RESPONSE)) return false;
        if (res==old_value) {
            wait_ms(1);
            if (i++>25) break;
        }
    } while(res==old_value);
    return res!=old_value;    
}


bool SI114x::nop(void) {
    return command(CMD_NOP);
}

bool SI114x::ps_force(void) {
    return command(CMD_PS_FORCE);
}

bool SI114x::als_force(void) {
    return command(CMD_ALS_FORCE);
}

bool SI114x::ps_als_force(void) {
    return command(CMD_PSALS_FORCE);
}

bool SI114x::ps_als_auto(void) {
    return command(CMD_PSALS_AUTO);
}

bool SI114x::als_auto(void) {
    return command(CMD_ALS_AUTO);
}

bool SI114x::read_param(uint8_t& val, uint8_t param) {
    uint8_t res = 0x00;
    if (!command(CMD_PARAM_QUERY|(param&0x1F))) return false;
    if (!read8(res,REG_PARAM_RD)) return false;
    return true;
}

bool SI114x::write_param(uint8_t param, uint8_t val, uint8_t* response) {
    if (!wait_until_sleep()) return false;
    uint8_t res = 0x00;
    uint8_t retval = 0x00;
    if (!read8(res,REG_RESPONSE)) return false;
    uint8_t buf[2] = { val, CMD_PARAM_SET|(param&0x1F) };
    if (!burst_write(REG_PARAM_WR,buf,2)) return false;
    do {
        if (!read8(retval,REG_RESPONSE)) return false;
        if (res==retval) wait_ms(1);
    } while(res==retval);
    if (response) *response = retval;
    return true;
}

bool SI114x::x_write_param(uint8_t param, uint8_t val, uint8_t* response) {
    write8(REG_PARAM_WR, val);
    x_command(CMD_PARAM_SET | param,response);
    return true;
}


bool SI114x::ps_als_pause(void) {
    return command(CMD_PSALS_PAUSE);
}

bool SI114x::pause_all(void) {
    uint8_t res = 0x00;
    while(1) {
        // keep sending nops until the response is zero        
        while(1) {
            if (!read8(res,REG_RESPONSE)) return false;
            if (res==0) {
                break;
            } else {
                if (!nop()) return false;
            }            
        }
        
        // pause the device        
        ps_als_pause();
        
        // wait for response, we expect a '1'
        uint8_t tries = 25;
        while(tries>0) {
            if (!read8(res,REG_RESPONSE)) return false;            
            if (res==1) return true;
            tries--;
            wait_ms(1);
        }

        // otherwise, try again        
    }
}

uint16_t SI114x::uncompress(uint8_t input) {
    if (input<8) return 0;
    uint8_t exponent = input>>4;
    uint16_t result = 0x10 | (input&0x0F);
    if (exponent>=4) return (result<<(exponent-4));
    return (result>>(4-exponent));
}

uint8_t SI114x::compress(uint16_t input) {
    if (input==0) return 0;
    if (input==1) return 8;
    uint16_t exponent = 0;
    uint16_t tmp = input;
    while(1) {
        tmp>>=1;
        exponent++;
        if (tmp==1) break;     
    }
    uint16_t mantissa = 0;
    if (exponent<5) {
        mantissa = (input<<(4-exponent));
        return ((exponent<<4)|(mantissa&0xF));
    }
    mantissa = input>>(exponent-5);
    if (mantissa&1) {
        mantissa+=2;
        if (mantissa&0x40) {
            exponent++;
            mantissa>>=1;
        }
    }
    return ((exponent<<4)|((mantissa>>1)&0xF));
}

bool SI114x::read8(uint8_t& val, uint8_t reg) {
    if (m_pI2C==NULL) return false;
    uint8_t ok = 0x00;
    m_pI2C->start();
        ok = (ok<<1)|m_pI2C->write(m_address);
        ok = (ok<<1)|m_pI2C->write(reg);
        m_pI2C->start();
        ok = (ok<<1)|m_pI2C->write(m_address|1);
        val = m_pI2C->read(0);
    m_pI2C->stop();
    return ok==0x7;
}

bool SI114x::x_read8(uint8_t& val, uint8_t reg) {
    m_pI2C->write(m_address, (char*)&reg, 1, true);
    m_pI2C->read(m_address, (char*)&val, 1);
    return true;
}

bool SI114x::read16(uint16_t& val, uint8_t reg) {
    if (m_pI2C==NULL) return false;
    uint8_t ok =0x00;
    m_pI2C->start();
        ok = (ok<<1)|m_pI2C->write(m_address);
        ok = (ok<<1)|m_pI2C->write(reg);
        m_pI2C->start();
        ok = (ok<<1)|m_pI2C->write(m_address|1);
        val = m_pI2C->read(1); // lsb
        val |= (uint16_t(m_pI2C->read(0))<<8);
    m_pI2C->stop();
    return ok==0x7;
}

bool SI114x::write8(uint8_t reg, uint8_t val) {
    if (m_pI2C==NULL) return false;
    uint8_t ok = 0x00;
    m_pI2C->start();
        ok = (ok<<1)|m_pI2C->write(m_address);
        ok = (ok<<1)|m_pI2C->write(reg);
        ok = (ok<<1)|m_pI2C->write(val);
    m_pI2C->stop();
    return ok == 0x7;
}

bool SI114x::x_write8(uint8_t reg, uint8_t val) {
    char data[] = {reg, val};
    m_pI2C->write(m_address, data, 2);
    return true;
}

bool SI114x::burst_read(uint8_t* data, uint8_t size, uint8_t reg) {
    if (m_pI2C==NULL) return false;
    if (size==0) return true;
    if (data==NULL) return false;
    uint8_t ok = 0x00;
    m_pI2C->start();
        ok = (ok<<1)|m_pI2C->write(m_address);
        ok = (ok<<1)|m_pI2C->write(reg);
        m_pI2C->start();
        ok = (ok<<1)|m_pI2C->write(m_address|1);
        for (uint8_t n=0; n<size-1; n++) data[n] = m_pI2C->read(1);
        data[size-1] = m_pI2C->read(0);
    m_pI2C->stop();
    return ok==0x7;
}

bool SI114x::burst_write(uint8_t reg, uint8_t* data, uint8_t size) {
    if (m_pI2C==NULL) return false;
    if (size==0) return true;
    if (data==NULL) return false;
    uint8_t ok = 0x00;
    m_pI2C->start();
        ok = (ok<<1)|m_pI2C->write(m_address);
        ok = (ok<<1)|m_pI2C->write(reg);
        for (uint8_t n=0; n<size; n++) ok+=m_pI2C->write(data[n]);
    m_pI2C->stop();
    return ok==3+size;
}

uint8_t SI114x::part_id(void) {
    if (m_pI2C==NULL) return false;
    uint8_t val = 0x00;
    if (!read8(val,REG_PART_ID)) return 0xFF;
    return val;
}

uint8_t SI114x::rev_id(void) {
    if (m_pI2C==NULL) return false;
    uint8_t val = 0x00;
    if (!read8(val,REG_REV_ID)) return 0xFF;
    return val;
}

uint8_t SI114x::seq_id(void) {
    if (m_pI2C==NULL) return false;
    uint8_t val = 0x00;
    if (!read8(val,REG_SEQ_ID)) return 0xFF;
    return val;
}

const SI114x::calib_t& SI114x::calib(void) const {
    return m_calib;
}

// ============================== CALIBRATION CODE ==============================

const uint32_t SI114x::fx20_bad_value = 0xFFFFFFFF;
const uint32_t SI114x::fx20_one       = SI114x::flt_to_fx20(1.0f);

uint32_t SI114x::flt_to_fx20(float x) {
    return ((x*1048576.0f)+0.5f);
}

void SI114x::fx20_round(uint32_t& value, int8_t bits) {
    uint32_t mask1 = ((2<<(bits  ))-1) << (31-(bits  ));
    uint32_t mask2 = ((2<<(bits-1))-1) << (31-(bits-1));
    uint32_t lsb   = mask1-mask2;
    int8_t shift = align(value,align_left);
    if ( (value&mask1)==mask1 ) {
        value = 0x80000000;
        shift-=1;
    } else {
        value+=lsb;
        value&=mask2;
    }
    shift_left(value,-shift);
}

uint32_t SI114x::fx20_div(uint32_t op1, uint32_t op2) {
    if ((op1==fx20_bad_value) || (op2==fx20_bad_value) || (op2==0)) return fx20_bad_value;
    fx20_round(op1);
    fx20_round(op2);
    int8_t numerator_sh   = align(op1,align_left);
    int8_t denominator_sh = align(op2,align_right);
    uint32_t result = op1 / uint16_t(op2);
    shift_left(result,20-numerator_sh-denominator_sh);
    return result;
}

uint32_t SI114x::fx20_mul(uint32_t op1, uint32_t op2) {
    fx20_round(op1);
    fx20_round(op2);
    int8_t op1_sh = align(op1,align_right);
    int8_t op2_sh = align(op2,align_right);
    
    uint32_t result = op1*op2;
    shift_left(result,-20+op1_sh+op2_sh);
    return result;
}


uint32_t SI114x::decode(uint32_t input) {
    if (input==0) return 0;    
    int32_t  exponent = ((input & 0x0F00) >> 8) - 9;
    uint32_t mantissa = 0x100|(input & 0xFF);
    mantissa<<=12+exponent;
    return mantissa;
}

uint32_t SI114x::collect(uint8_t* buffer, uint8_t msb_addr, uint8_t lsb_addr, uint8_t align) {
    uint16_t val;
    uint8_t msb = *(buffer+msb_addr-0x22);
    uint8_t lsb = *(buffer+lsb_addr-0x22);
    if (align==0) {
        val = (uint16_t(msb)<<4)|(uint16_t(lsb)>>4);
    } else {
        val = ((uint16_t(msb)&0xF)<<8)|uint16_t(lsb);
    }
    if ((val==0xFFF) || (val==0) ) return fx20_bad_value;
    return decode(val);
}

void SI114x::shift_left(uint32_t& value, int8_t shift) {
    if (shift>0) value<<=shift;
    else value>>=-shift;
}

const int8_t SI114x::align_left = 1;
const int8_t SI114x::align_right=-1;

int8_t SI114x::align(uint32_t& value, int8_t dir) {
    if (value==0) return 0;
    int8_t local_shift, shift;
    uint32_t mask;
    switch(dir) {
        case align_left:
            local_shift = 1;
            mask = 0x80000000;
            break;
        case align_right:
            local_shift=-1;
            mask = 0x1;
            break;
        default:
            return 0;
    }
    shift = 0;
    while(1) {
        if (value&mask) break;
        shift++;
        shift_left(value,local_shift);
    }
    return shift;
}

const SI114x::refcalib_t SI114x::m_refcalib[2] = {
    { 
        flt_to_fx20( 4.021290),  // sirpd_adchi_irled
        flt_to_fx20(57.528500),  // sirpd_adclo_irled
        flt_to_fx20( 2.690010),  // sirpd_adclo_whled
        flt_to_fx20( 0.042903),  // vispd_adchi_whled
        flt_to_fx20( 0.633435),  // vispd_adclo_whled
        flt_to_fx20(23.902900),  // lirpd_adchi_irled
        flt_to_fx20(56.889300),  // ledi_65ma
        {0x7B,0x6B,0x01,0x00}    // default ucoef
    },
    {
        flt_to_fx20( 2.325484),  // sirpd_adchi_irled
        flt_to_fx20(33.541500),  // sirpd_adclo_irled
        flt_to_fx20( 1.693750),  // sirpd_adclo_whled
        flt_to_fx20( 0.026775),  // vispd_adchi_whled
        flt_to_fx20( 0.398443),  // vispd_adclo_whled
        flt_to_fx20(12.190900),  // lirpd_adchi_irled
        flt_to_fx20(56.558200),  // ledi_65ma
        {0xDB,0x8F,0x01,0x00}    // default ucoef
    }
};

bool SI114x::read_calibration(void) {
    m_calib.vispd_correction = fx20_one;
    m_calib.irpd_correction  = fx20_one;
    m_calib.adcrange_ratio   = flt_to_fx20(14.5);
    m_calib.irsize_ratio     = flt_to_fx20( 6.0);
    m_calib.ledi_ratio       = fx20_one;
    m_calib.ucoef            = NULL;
    
    // 1. make sure that the device is ready to receive commands
    uint8_t res = 0x00;
    do {
        if (!nop()) return false;
        if (!read8(res,REG_RESPONSE)) return false;
    } while(res!=0);
    
    // 2. request the calibration data
    if (!command(CMD_GET_CAL,&res)) return false;
    if (!wait_until_sleep()) return false;
    
    // 3. wait for response register to increment
    do {
        if (!read8(res,REG_RESPONSE)) return false;
        if (res==0x80) {
            // calibration code is not implemented on this device
            nop();
            return false;
        } else {
            if (res&0xF0) return false;
        }    
    } while (res!=1);
    
    // 4. retrieve 12 bytes from interface registers    
    uint8_t buffer[14];
    if (!burst_read(buffer,12,REG_ALS_VIS_DATA0)) return false;
    if (!get_calibration_index(buffer)) return false;
    int16_t index = cal_index(buffer);
    
    m_calib.vispd_correction = vispd_correction(buffer);
    m_calib.irpd_correction  = irpd_correction(buffer);
    m_calib.adcrange_ratio   = adcrange_ratio(buffer);
    m_calib.irsize_ratio     = irsize_ratio(buffer);
    m_calib.ledi_ratio       = ledi_ratio(buffer);
    if (index>=0)   m_calib.ucoef = m_refcalib[index].ucoef;
    else            m_calib.ucoef = m_refcalib[0].ucoef;    
    return true;
}

bool SI114x::get_calibration_index(uint8_t* buffer) {
    if (buffer==NULL) return false;
    uint8_t res = 0x00;
    
    // 1. make sure device is ready to receive commands
    do {
        if (!nop()) return false;
        if (!read8(res,REG_RESPONSE)) return false;
    } while (res!=0);
    
    // 2. retrieve the index
    if (!write8(REG_COMMAND,CMD_GET_CAL_INDEX)) return false;
    if (!wait_until_sleep()) return false;
    
    // 3. get calibration index
    if (!burst_read(buffer+12,2,REG_PS1_DATA0)) return false;
    
    return true;
}

int16_t SI114x::cal_index(uint8_t* buffer) {
    int16_t index = uint16_t(buffer[12]) | (uint16_t(buffer[13])<<8);    
    switch(index) {
        case -1: index = 0; break;
        default: index = -(2+index);
    }    
    uint8_t size = sizeof(m_refcalib)/sizeof(m_refcalib[0]);
    if (index<size) return index;
    return -1;
}

uint32_t SI114x::_SIRPD_ADCHI_IRLED(uint8_t* buffer) {
    return collect(buffer, 0x23, 0x22,  0);
}

uint32_t SI114x::_SIRPD_ADCLO_IRLED(uint8_t* buffer) {
    return collect(buffer, 0x22, 0x25,  1);
}

uint32_t SI114x::_SIRPD_ADCLO_WHLED(uint8_t* buffer) {
    return collect(buffer, 0x24, 0x26,  0);
}

uint32_t SI114x::_VISPD_ADCHI_WHLED(uint8_t* buffer) {
    return collect(buffer, 0x26, 0x27,  1);
}

uint32_t SI114x::_VISPD_ADCLO_WHLED(uint8_t* buffer) {
    return collect(buffer, 0x28, 0x29,  0);
}

uint32_t SI114x::_LIRPD_ADCHI_IRLED(uint8_t* buffer) {
    return collect(buffer, 0x29, 0x2a,  1);
}

uint32_t SI114x::_LED_DRV65(uint8_t* buffer) {
    return collect(buffer, 0x2b, 0x2c,  0);
}


uint32_t SI114x::vispd_correction(uint8_t* buffer) {
    int16_t index = cal_index(buffer);
    if (index<0) return fx20_one;
    uint32_t result = fx20_div(m_refcalib[index].vispd_adclo_whled,_VISPD_ADCLO_WHLED(buffer));
    if (result==fx20_bad_value) return fx20_one;
    return result;
}

uint32_t SI114x::irpd_correction(uint8_t* buffer) {
    int16_t index = cal_index(buffer);
    if (index<0) return fx20_one;
    uint32_t result = fx20_div(m_refcalib[index].sirpd_adclo_irled,_SIRPD_ADCLO_IRLED(buffer));
    if (result==fx20_bad_value) return fx20_one;
    return result;
}

uint32_t SI114x::adcrange_ratio(uint8_t* buffer) {
    uint32_t result = fx20_div(_SIRPD_ADCLO_IRLED(buffer),_SIRPD_ADCHI_IRLED(buffer));
    if (result==fx20_bad_value) return flt_to_fx20(14.5f);
    return result;
}

uint32_t SI114x::irsize_ratio(uint8_t* buffer) {
    uint32_t result = fx20_div(_LIRPD_ADCHI_IRLED(buffer),_SIRPD_ADCHI_IRLED(buffer));
    if (result==fx20_bad_value) return flt_to_fx20(6.0f);
    return result;
}

uint32_t SI114x::ledi_ratio(uint8_t* buffer) {
    int16_t index = cal_index(buffer);
    if (index<0) return fx20_one;
    uint32_t result = fx20_div(m_refcalib[index].ledi_65ma,_LED_DRV65(buffer));
    if (result==fx20_bad_value) return fx20_one;
    return result;
}
