/** 
*   Temperature/humidity/pressure broadcast example for mbed / VTT Node V3   
*   o   Using BMP180 & Si7021 and onboard temperature sensor
*   Juho Eskeli, VTT
*/


#include "NodeV3PinNames.h" //This should come before mbed.h to override pin configuration
#include "mbed.h"
#include "ble/BLE.h"

#define USE_DFU

#ifdef USE_DFU
#include "DFUService.h"
#endif

#include "AT45.h"
#include "TMP_nrf51.h"

#include "BMP180.h"
#include "Si7021.h"

//interrupt /gpio configuration
#include "nrf_gpio.h"
#include "nrf_gpiote.h"
#include "nrf_soc.h"

//const static char     DEVICE_NAME[]        = "BAc0N";
//static const uint16_t uuid16_list[]        = {GattService::UUID_STREAMING_SERVICE};

BLE        ble;

#define MOSI SPI_PSELMOSI0
#define MISO SPI_PSELMISO0
#define CS SPI_PSELSS0
#define SCLK SPI_PSELSCK0

static TMP_nrf51  tempSensor;   
BMP180 bmp180(I2C_SDA0, I2C_SCL0);
SI7021_I2C si7021(I2C_SDA0, I2C_SCL0);                
                                                                                            
DigitalOut myled1(LED1);
DigitalOut myled2(LED2);

DigitalOut LIS_CS_0(CS); //LIS3DH CS
DigitalOut AT_CS(p5); //Dataflash CS
DigitalOut AT_RS(p6); //Dataflash reset

__packed struct ApplicationData_t {
    uint16_t                    applicationSpecificId; /* An ID used to identify temperature value in the manufacture specific AD data field */
    TMP_nrf51::tmpSensorValue_t tmpSensorValue;      /* User defined application data */           
    int16_t                     bmp180_temp;
    uint32_t                    bmp180_press;
    float                       si7021_temperature;
    float                       si7021_humidity;
};

void setupApplicationData(ApplicationData_t &appData)
{
    static const uint16_t APP_SPECIFIC_ID_TEST = 0xFEFE;
    appData.applicationSpecificId = APP_SPECIFIC_ID_TEST;
    appData.tmpSensorValue        = tempSensor.get();          
    appData.bmp180_temp           = bmp180.BMP180GetTemperature();
    appData.bmp180_press          = bmp180.BMP180GetPressure();
    si7021.Measure_Humidity_Temp((float*)&(appData.si7021_humidity), (float*)&(appData.si7021_temperature));
}
 
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    ble.gap().startAdvertising();
}

void senseCallback(void)
{   
    ApplicationData_t appData;  
    setupApplicationData(appData);
    ble.gap().updateAdvertisingPayload(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA, (uint8_t *)&appData, sizeof(ApplicationData_t));             
}

void setup_CS_LIS3DH()
{
    //Without this setup CS lines of AT45 & LIS3DH are in conflict, causing huge current consumption    
    LIS_CS_0 = 1; //not selected
    AT_CS = 1; //not selected
    AT_RS = 0; //asserted == reset state
    wait_ms(100);
    AT_RS = 1;
}

bool init_bmp180(void) 
{    
    // Read the WHO_AM_I register of the BMP-180, this is a good test of communication
    uint8_t c = bmp180.readByte(BMP180_ADDRESS, BMP180_WHO_AM_I);   
    if(c == 0x55) {
        //BMP-180 should be 0x55
        //BMP-180 online...
        bmp180.BMP180Calibration();
        //BMP-180 calibration complete        
        return true;
    }
    else 
    {                
        return false;
    }
}

int main() 
{    
    //LEDS off
    myled1 = 0;
    myled2 = 0;
    
    setup_CS_LIS3DH();
    
    //Initialize SPI interface
    SPI spi(MOSI, MISO, SCLK); // mosi, miso, sclk
    spi.format(8,3);
    spi.frequency(8000000); 
    //setup AT45 dataflash to powersaving mode
    AT45* flash = new AT45(spi, p5);    
    flash->ultra_deep_power_down(true);     
    
    init_bmp180();
    
    //Initialize BLE stuff
    ble.init();
    ble.gap().onDisconnection(disconnectionCallback);
    
    #ifdef USE_DFU
    DFUService dfu(ble);
    #endif
    
    /* Setup advertising. */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_THERMOMETER);
    ApplicationData_t appData;
    setupApplicationData(appData);
    //ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    //ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA, (uint8_t *)&appData, sizeof(ApplicationData_t));
    
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    //ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_NON_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(100); /* 1000ms. */
    ble.gap().startAdvertising();
    
    Ticker ticker;
    ticker.attach(senseCallback, 0.1);
  
    ///Could have main loop here doing something
    while(true) {
        ble.waitForEvent();
   }     
}
 
