/* NetworkSocketAPI Example Program
 * Copyright (c) 2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "WizFi310Interface.h"
#include "TCPSocket.h"

#if defined(TARGET_NUCLEO_F401RE)
Serial pc(USBTX,USBRX);
WizFi310Interface wifi(PA_11, PA_12, D6, D7, D3, NC, 115200);
#endif

AnalogIn sensor_illu(PA_1);
DigitalOut led(PC_7);
InterruptIn button(PC_13);


#define AP_SSID "<AP SSID>"
#define AP_PASSWORD "<AP Password>"
#define AP_SECURITY NSAPI_SECURITY_WPA2

#define IoTMakers_GW_ID "<Gateway ID>"
#define IoTMakers_DEV_ID "<Device ID>"
#define IoTMakers_DEV_PWD "<Device Password>"

#define IoTMakers_MQTT_IP "220.90.216.90"
#define IoTMakers_MQTT_PORT 10030


#define DELAY_SEND_TIME_MS 5000

unsigned long g_time_ms = 0;
unsigned long g_prev_send_time = 0;
unsigned long g_prev_clicked_time = 0;

//Button IRQ
int button_status = 0;
int button_clicked = 0;


enum Connection_Step {
    csDISASSOCIATE = 0,
    csASSOCIATE,
    csKTCONNECTED,
}CON_STEP;

TCPSocket socket(&wifi);
WizFi310* pwizfi310 = wifi.get_WizFi310_Pointer();

void Connect_to_AP();
void Connect_to_IoTMakers();
void Send_num_value_to_IoTMakers(const char* tag_stream, const float dvalue);
void Send_str_value_to_IoTMakers(const char* tag_stream, const char* svalue);

void Button_IRQ();
void Button_Push_Event();
void Periodic_Event();

void time_ms();

void Button_IRQ()
{
    if( button_clicked == 0 && (g_prev_clicked_time + 500) < g_time_ms)
    {
        button_clicked = 1;
        g_prev_clicked_time = g_time_ms;
    }
}
void Button_Push_Event()
{

    if( CON_STEP == csKTCONNECTED)
    {
        if( button_status == 0 )
        {  
            Send_str_value_to_IoTMakers("switch", "on");
            button_status = 1;
        }
        else if( button_status == 1 )
        {
            Send_str_value_to_IoTMakers("switch", "off");
            button_status = 0;
        }
    }
}

void Periodic_Event()
{
    float illumination;
    if( CON_STEP == csKTCONNECTED)
    {
        illumination = sensor_illu.read() * 100;
        
        Send_num_value_to_IoTMakers("illumination", (float)illumination);
    }
}


int main()
{
    pc.baud(115200);
    
    //Timer settings
    Ticker timer_ms;
    timer_ms.attach(time_ms, 0.001f);
    g_prev_send_time = g_time_ms;
    
    //button Interrupt
    button.rise(&Button_IRQ);
    
    
    //normal settings
    CON_STEP = csDISASSOCIATE;

    printf("KT IoTMakers Example\r\n");
    
    char buffer[256];
    int count;
   
   while(1)
   { 
        switch (CON_STEP)
        {
            case csDISASSOCIATE:
                Connect_to_AP();
                break;
            case csASSOCIATE:
                Connect_to_IoTMakers();
                break;
         
            case csKTCONNECTED:
                
                if( ( g_prev_send_time + DELAY_SEND_TIME_MS ) < g_time_ms )
                {  
                    Periodic_Event();
                    g_prev_send_time = g_time_ms;
                }
                
                if( button_clicked == 1 )
                {
                    Button_Push_Event();     
                    button_clicked = 0;
                }
                
                if( pwizfi310->readable(0) > 0)
                {         
                    pwizfi310->recv(0, buffer, sizeof(buffer));
                    printf("RECV : %s\r\n", buffer);
                }
                
                
                break;
            default:
                break;
        };
    }

    


}


void Connect_to_AP()
{
    int i;
    int ret;

    for( i=0; i<5; i++ )
    {
        
        if( CON_STEP != csDISASSOCIATE )
        {
            break;
        }
        
        ret = wifi.connect(AP_SSID, AP_PASSWORD, AP_SECURITY);
        
        if( ret == 0 ) //connection success
        {
            //const char *ip = wifi.get_ip_address();
            //const char *mac = wifi.get_mac_address();
            //printf("IP address is: %s\r\n", ip ? ip : "No IP");
            //printf("MAC address is: %s\r\n", mac ? mac : "No MAC");
            
            CON_STEP = csASSOCIATE;
            wait(1);
            break;           
        }
        else
        {
            printf("CONNECTION FAIL\r\n", ret);
            CON_STEP = csASSOCIATE;
        }
    }
}

void Connect_to_IoTMakers()
{
    char cmd[100];
    int ret = 0;
    
    //set_information
    sprintf(cmd, "AT+TKTSET1=%s,%s,%s,0", IoTMakers_GW_ID, IoTMakers_DEV_ID, IoTMakers_DEV_PWD);
    pwizfi310->sendCommand(cmd);
    
    //connect IoTMakers via MQTT
    sprintf(cmd, "AT+TKTCON=2,%s,%d,0,1111", IoTMakers_MQTT_IP, IoTMakers_MQTT_PORT);
    ret = pwizfi310->sendCommand(cmd,WizFi310::RES_NULL,5000);
    
    if(ret == 0)
    {        
        CON_STEP = csKTCONNECTED;
    }
}

void Send_num_value_to_IoTMakers(const char* tag_stream, const float dvalue)
{
    char cmd[100];
    //send numberic data
    sprintf(cmd, "AT+TKTSEND=d,%s,%.1f", tag_stream, dvalue);
    pwizfi310->sendCommand(cmd);
}

void Send_str_value_to_IoTMakers(const char* tag_stream, const char* svalue)
{
    char cmd[100];
    //send string data
    sprintf(cmd, "AT+TKTSEND=s,%s,%s", tag_stream, svalue);
    pwizfi310->sendCommand(cmd);
}


void time_ms()
{
     g_time_ms++;   
}