/*
** This software can be freely used, even comercially, as highlighted in the license.
** 
** Copyright 2014 GHI Electronics, LLC
** 
** Licensed under the Apache License, Version 2.0 (the "License");
** you may not use this file except in compliance with the License.
** You may obtain a copy of the License at
** 
**     http://www.apache.org/licenses/LICENSE-2.0
** 
** Unless required by applicable law or agreed to in writing, software
** distributed under the License is distributed on an "AS IS" BASIS,
** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
** See the License for the specific language governing permissions and
** limitations under the License.
**
**/

#pragma once
#include <string>
#include "mbed.h"


class mBot
{

public:  // constructors, enums
	mBot();

	
	typedef enum
	{
		/// <summary>
		/// Represents the left sensor.
		/// </summary>
		Left,

		/// <summary>
		/// Represents the right sensor.
		/// </summary>
		Right
	} ReflectiveSensors;	
	
private: // fields
	static const float MOTOR_BASE_PERIOD = 0.00001;  // 100KHz

	static PwmOut *leftMotor;
	static PwmOut *rightMotor;
	
	static DigitalOut *leftMotorDirectionCtrl_1;
	static DigitalOut *leftMotorDirectionCtrl_2;
	
	static DigitalOut *rightMotorDirectionCtrl_1;
	static DigitalOut *rightMotorDirectionCtrl_2;

	static bool leftMotorInverted;
	static bool rightMotorInverted;
	
	static AnalogIn *leftSensor;
	static AnalogIn *rightSensor;
	static DigitalOut *leftIRLED;
	static DigitalOut *rightIRLED;
	

public:  // methods ....

	/// <summary>
	/// Gets the reading from a reflective sensor between 0 and 1024 (10 bit resolution). The higher the number, 
	/// the more reflection that was detected. Nearby objects reflect more than far objects.
	/// </summary>
	/// <param name="sensor">The sensor to read from.</param>
	static float GetReflectiveReading(ReflectiveSensors sensor);

	/// <summary>
	/// Turns the reflective sensors on or off to save power (1 = on, 0 = off). They are on by default.
	/// </summary>
	static  void SetReflectiveSensorState(int state);


	/// <summary>
	/// The reflective sensors on the mBot.
	/// </summary>
	/// <summary>
	/// Sets the period and duration that the buzzer will buzz for.
	/// </summary>
	/// <param name="period">The period that the buzzer will buzz in hertz.</param>
	/// <param name="duration">The duration the buzzer will buzz for in milliseconds.</param>
	/// <param name="dutyCycle">The duty cycle for the buzzer.</param>
	/// <remarks>If duration is 0, the buzzer will buzz indefinitely. If it is non-zero, then mBot call will block for as many milliseconds as specified in duration, then return.</remarks>

	/// <summary>
	/// Sets the speed of the motor. -100 is full speed backwards, 100 is full speed forward, and 0 is stopped.
	/// </summary>
	/// <param name="leftSpeed">The new speed of the left motor.</param>
	/// <param name="rightSpeed">The new speed of the right motor.</param>
	static void SetMotorSpeed(int leftSpeed, int rightSpeed);
	
	/// <summary>
	/// Change the wheel directions (useful if you want to treat the batteries as the back of the robot and the wheels as the front).
	/// </summary>
	/// <param name="invertLeft">true then inverted.</param>
	/// <param name="invertRight">true then inverted.</param>
	static void SetMotorInversion(bool invertLeft, bool invertRight);
	

private:
	static void SetSpeed(PwmOut *motor, DigitalOut *directionCtrl_1,  DigitalOut *directionCtrl_2, int speed);  //, bool isLeft);
};

