/* This library encapsulates (or will when completely implemented)
 * all software functions for reading and writing to the Cypress Semiconductor
 * CY14B101P nvSRAM module.
 *
 * As of July 11, 2011 RTC functions are not implemented.
 *
 * Author: Dr. Jeffrey Craighead
 * Copyright 2011
 */


#ifndef CY14B101P
#define CY14B101P

/*****
* Includes
*****/
#include "mbed.h"


/*****
* Status Register Instructions
*****/
#define WREN 0x06 //Set Write Enable latch
#define WRDI 0x04 //Reset Write Enable latch
#define RDSR 0x05 //Read Status Register
#define WRSR 0x01 //Write Status Register

/*****
* SRAM Instructions
*****/
#define READ 0x03 //Read data from memory array
#define WRITE 0x02 //Write data to memory array

/*****
* RTC Instructions
*****/
#define WRTC 0x12 //Write Real Time Clock registers
#define RDRTC 0x13 //Read Real Time Clock registers

/*****
* Special NV Instructions
*****/
#define STORE 0x3C //Software Store (to NV memory)
#define RECALL 0x60 //Software Recall (from NV memory)
#define ASENB 0x59 //AutoStore Enable
#define ASDISB 0x19 //AutoStore Disable

/*****
* Other Definitions
*****/
#define MAXADDR 0x1FFFF

/*****
* Status Register Bits
*****/
#define READY 0x00 //Read Only, Active Low, bit is 1 while a STORE or RECALL is in progress
#define WENABLE 0x01 //Write enabled bit
#define BP0 0x02; //Block protection bit 0
#define BP1 0x03; //Block protection bit 1
#define WPENABLE 0x07; //Enabled the use of the hardware Write Protect pin.



/** Example:
 * @code
 * #include "CY14B101P.h"
 * #include "mbed.h"
 *
 * NVSRAM mem(p11,p12,p13,p14,40000000,8,0);
 * Serial pc(USBTX,USBRX);
 *
 * int main() {
 *    static char myString[] = "A Test String!\r\n";
 *    char readString[sizeof(myString)];
 *
 *    Timer s; //This timer is just to monitor read & write speed
 *    s.start();
 *    s.reset();
 *
 *    mem.init();   
 *    while (true) {
 *        int start = s.read_us();
 *        //Write bytes
 *        mem.writeBytes(myString,0,sizeof(myString));
 *        int writeEnd = s.read_us();
 *
 *        //Read bytes
 *        mem.readBytes(readString,0,sizeof(readString));
 *        int end = s.read_us();
 *
 *        //Print read bytes to Serial to verify correctness
 *        for (int i=0; i<sizeof(readString); i++) pc.printf("%c",readString[i]);
 *        pc.printf("\r\n");
 *        pc.printf("Write completed in %d microsecs.\r\n",writeEnd-start);
 *        pc.printf("Read completed in %d microsecs.\r\n",end-writeEnd);
 *        pc.printf("Status byte: 0x%x\r\n",mem.readStatusRegister());
 *        wait(0.5);
 *    }
 *}
 *@endcode
 */


/*****
* Cypress CY14B101P SPI nvSRAM Memory
*****/

class NVSRAM {

public:

    /**Constructor - Create an NVRAM instance 
     * @param mosi Pin to use as SPI mosi
     * @param miso Pin to use as SPI miso
     * @param sclk Pin to use as SPI clock
     * @param csel Pin to use as SPI chip select
     * @param spifrequency Frequency of the SPI bus, 40MHz max
     */
    NVSRAM(PinName mosi, PinName miso, PinName sclk, PinName csel, int spifrequency);

    
    /**Initialize the nvRAM status register allowing writes across the entire memory range.
     *
     */
    void init();

    /** Write bytes to the SRAM
     * @param bytes A pointer to an array of characters containing the data to be written
     * @param address The starting address in the SRAM for the write
     * @param length The number of bytes to write
     */
    void writeBytes(char *bytes, unsigned int address, int length);

    /** Read bytes frin the SRAM
     * @param bytes A pointer to an array of characters that will be written to
     * @param address The starting address in the SRAM for the read
     * @param length The number of bytes to read
     */
    void readBytes(char *bytes, unsigned int address, int length);

    //Set the Real Time Clock
    void setRTC(int century, int year, int month, int dayofmonth, int dayofweek, int hour, int minute, int second);

    //Read the RTC
    int readRTC();

    /** Store the contents of the SRAM to the NV memory
     *
     */
    void nvStore();

    /** Restore the contents of the SRAM from the NV memory
     *
     */
    void nvRecall();

    /** Boolean to enable the AutoStore feature of the CY14B101P
     * When enabled, the contents of the SRAM will automatically be stored
     * to the NV memory on poweroff. Requires a capacitor between Vcap and Vss
     *
     * @param enable True if you want to enable autostore, False if you want to disable autostore
     */
     void enableAutoStore(bool enable);
     

    /** Write to the status register
     * @param status The byte to write to the status register
     */
    void writeStatusRegister(char status);
    
    /** Read the status register
     * @param returns The byte contained in the status register
     */
    char readStatusRegister();

private:
    int spifreq;
    SPI spi_;
    DigitalOut chipSel_;
};

#endif /* CY14B101P */