/* This library encapsulates (or will when completely implemented)
 * all software functions for reading and writing to the Cypress Semiconductor
 * CY14B101P nvSRAM module.
 *
 * As of July 11, 2011 RTC functions are not implemented.
 *
 * Author: Dr. Jeffrey Craighead
 * Copyright 2011
 */

/** Example:
 * @code
 * #include "CY14B101P.h"
 * #include "mbed.h"
 *
 * NVSRAM mem(p11,p12,p13,p14,40000000,8,0);
 * Serial pc(USBTX,USBRX);
 *
 * int main() {
 *    static char myString[] = "A Test String!\r\n";
 *    char readString[sizeof(myString)];
 *
 *    Timer s; //This timer is just to monitor read & write speed
 *    s.start();
 *    s.reset();
 *
 *    mem.init();   
 *    while (true) {
 *        int start = s.read_us();
 *        //Write bytes
 *        mem.writeBytes(myString,0,sizeof(myString));
 *        int writeEnd = s.read_us();
 *
 *        //Read bytes
 *        mem.readBytes(readString,0,sizeof(readString));
 *        int end = s.read_us();
 *
 *        //Print read bytes to Serial to verify correctness
 *        for (int i=0; i<sizeof(readString); i++) pc.printf("%c",readString[i]);
 *        pc.printf("\r\n");
 *        pc.printf("Write completed in %d microsecs.\r\n",writeEnd-start);
 *        pc.printf("Read completed in %d microsecs.\r\n",end-writeEnd);
 *        pc.printf("Status byte: 0x%x\r\n",mem.readStatusRegister());
 *        wait(0.5);
 *    }
 *}
 *@endcode
 */




#include "CY14B101P.h"

Serial pc_(USBTX,USBRX);
//Constructor - specify mbed pins for SPI connection to memory and the SPI clock rate
NVSRAM::NVSRAM(PinName mosi, PinName miso, PinName sclk, PinName csel, int spifrequency) : spi_(mosi, miso, sclk), chipSel_(csel) {

    //chipSel_(csel);

    spi_.format(8,0);
    spi_.frequency(spifrequency);
    spifreq = spifrequency;
    chipSel_=1;
}


void NVSRAM::init(){
    writeStatusRegister(0x00);
}

void NVSRAM::writeBytes(char *bytes, unsigned int address, int length){
    chipSel_ = 0;
    spi_.write(WREN);
    chipSel_ = 1;
    
    chipSel_ = 0;
    
    spi_.write(WRITE);
   
    spi_.write( (unsigned char)((0x00010000 & address)>>16) );
    spi_.write( (unsigned char)((0x0000FF00 & address)>>8) );
    spi_.write( (unsigned char)(0x000000FF & address) );
    //pc_.printf("Writing To: %x%x%x\r\n",(0x00010000 & address)>>16,(0x0000FF00 & address)>>8,(0x000000FF & address));
    for(int i=0; i<length; i++) spi_.write(bytes[i]);

    chipSel_ = 1;
}

void NVSRAM::readBytes(char *bytes, unsigned int address, int length){
    chipSel_ = 0;

    spi_.write(READ);
    
    spi_.write( (unsigned char)((0x00010000 & address)>>16) );
    spi_.write( (unsigned char)((0x0000FF00 & address)>>8) );
    spi_.write( (unsigned char)(0x000000FF & address) );
    //pc_.printf("Reading From: %x%x%x\r\n",(0x00010000 & address)>>16,(0x0000FF00 & address)>>8,(0x000000FF & address));
    for(int i=0; i<length; i++) bytes[i] = spi_.write(0x00);

    chipSel_ = 1;
}

void NVSRAM::setRTC(int century, int year, int month, int dayofmonth, int dayofweek, int hour, int minute, int second){
    //RTC SPI frequency must be no greater than 25MHz, so just use half of the SRAM frequency since that has a max of 40MHz
    spi_.frequency(spifreq/2);
    chipSel_=0;
    spi_.write(WREN);
    chipSel_=1;
    
    
    //Set the SPI frequency back to the requested rate for NVRAM access.
    spi_.frequency(spifreq);
}

int NVSRAM::readRTC(){
    //RTC SPI frequency must be no greater than 25MHz, so just use half of the SRAM frequency since that has a max of 40MHz
    spi_.frequency(spifreq/2);
    
    
    
    //Set the SPI frequency back to the requested rate for NVRAM access.
    spi_.frequency(spifreq);


    return 0;
}

void NVSRAM::nvStore(){
    chipSel_=0;
    spi_.write(WREN);
    chipSel_=1;
    chipSel_=0;
    spi_.write(STORE);
    chipSel_=1;
}

void NVSRAM::nvRecall(){
    chipSel_=0;
    spi_.write(WREN);
    chipSel_=1;
    chipSel_=0;
    spi_.write(RECALL);
    chipSel_=1;
}

void NVSRAM::enableAutoStore(bool enable){
    chipSel_=0;
    spi_.write(WREN);
    chipSel_=1;
    chipSel_=0;
    if(enable) spi_.write(ASENB);
    else spi_.write(ASDISB);
    chipSel_=1;
}

void NVSRAM::writeStatusRegister(char status){
    chipSel_=0;
    spi_.write(WREN);
    chipSel_=1;
    chipSel_=0;
    spi_.write(WRSR);
    spi_.write(status);
    chipSel_=1;
}

char NVSRAM::readStatusRegister(){
    chipSel_ = 0;
    spi_.write(RDSR);
    char out = spi_.write(0x00);
    chipSel_ = 1;
    
    return out;
}