// Uses Timer to read servo pulse on single input pin
// J. Bradshaw 20140925
/* Copyright (c) 2015, jbradshaw (http://mbed.org)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
*/ 
#include "mbed.h"
#include "ServoIn.h"

ServoIn::ServoIn(PinName pin): _interrupt(pin), _pin(pin){        // create the InterruptIn on the pin specified to ServoIn
    _interrupt.rise(this, &ServoIn::PulseRead); // attach PulseRead function of this ServoIn instance
    
    _validPulseTimeout.attach(this, &ServoIn::timeoutTest, 0.015); // the member function, and interval (15ms)
    pulseMeasure.start();
    if(_pin)
        _pulseFlag = 1;
    else
        _pulseFlag = 0;
    servoPulse = 0;
    servoPulseMax = 2100;
    servoPulseMin = 900;
    _t_state=0;    
}

//function to read the servo pulse duration
void ServoIn::PulseRead() {
    if(!_pulseFlag){ //if the flag is low, was waiting for high input interrupt
        this->pulseMeasure.reset();   //reset the timer
        _interrupt.fall(this, &ServoIn::PulseRead);
        _pulseFlag = 1;  //set the pulse input flag
    }
    else{
        servoPulse = pulseMeasure.read_us();
        if(servoPulse < this->servoPulseMax){
            _interrupt.rise(this, &ServoIn::PulseRead);
            _pulseFlag = 0;
        }
        else{
            _pulseFlag = 1;  //set the pulse input flag
        }
    }
    _validPulseTimeout.attach(this, &ServoIn::timeoutTest, 0.015); //setup interrupt for 15ms
    _t_state=0;  //re-zero t-state    
}


int ServoIn::read(void){
    return servoPulse;    
}

float ServoIn::readCalPulse(void){
    servoPulseOffset = ((float)servoPulseMax + (float)servoPulseMin) / 2.0;
    servoPulseScale = 1.0 / (((float)servoPulseMax - (float)servoPulseMin) / 2.0);
    
    float pulseCal = ((float)servoPulse-servoPulseOffset)*servoPulseScale;
    if((pulseCal > 1.5) || (pulseCal < -1.5))
        return 0.0;
        
    return ((float)servoPulse-servoPulseOffset)*servoPulseScale;    
}

void ServoIn::timeoutTest(void){
    if(!_t_state){
        _validPulseTimeout.attach(this, &ServoIn::timeoutTest, 0.015);
        _t_state=1;  //toggle t_sate (15ms has elapsed without servo edge
    }
    else{   //30 ms has elapsed without a valid servo pulse
        _pulseFlag = 0;
        _t_state=0;
        servoPulse = 0;
        _validPulseTimeout.attach(this, &ServoIn::timeoutTest, 0.015);
    }
}

