/*
 * An example showing how to use the CANnucleo library:
 *
 * Two affordable (less than $4 on ebay) STM32F103C8T6 boards (20kB SRAM, 64kB Flash), 
 * compatible with the NUCLEO-F103RB platform (20kB SRAM, 128kB Flash), 
 * are connected to the same CAN bus via transceivers (MCP2551 or TJA1040, or etc.). 
 * CAN transceivers are not part of NUCLEO boards, therefore must be added by you. 
 * Remember also that CAN bus (even a short one) must be terminated with 120 Ohm resitors at both ends.
 *
 * For more details see the wiki page <https://developer.mbed.org/users/hudakz/code/CAN_Nucleo_Hello/>
 *
 * NOTE: If you'd like to use the official NUCLEO-F103RB boards
 *       comment out the line #define TARGET_STM32F103C8T6  1
 *
 * The same code is used for both NUCLEO boards, but:
 *      For board #1 compile the example without any change.
 *      For board #2 comment out the line #define BOARD1 1 before compiling 
 *
 * Once the binaries have been downloaded to the boards reset board #1.
 *
 */ 

#include "mbed.h"
#include "CAN.h"
#include "C12832.h"
#include "LM75B.h"

#define BOARD1    1                 // comment out this line when compiling for board #2

#if defined(BOARD1)
    #define RX_ID   0x100
    #define TX_ID   0x7FF
#else
    #define RX_ID   0x101
    #define TX_ID   0x7FF
#endif

// See wiki page <https://developer.mbed.org/users/hudakz/code/CAN_Nucleo_Hello/>
//#define TARGET_STM32F103C8T6  1     // comment out this line if you'd like to use the official NUCLEO-F103RB boards
                                    
#if defined(TARGET_STM32F103C8T6)  
    DigitalOut  led(PC_13);
#else
    DigitalOut  led(LED1);
#endif

    DigitalIn fire(D4);
    C12832 lcd(D11, D13, D12, D7, D10);
    LM75B sensor(D14,D15);
    AnalogIn pot1 (A0);
    AnalogIn pot2 (A1);
    PwmOut r (D5);
    PwmOut g (D8);
    PwmOut b (D9);
      
int             ledReceived;
Timer           timer;
CAN             can(PA_11, PA_12);  // CAN Rx pin name, CAN Tx pin name, Automatic recovery from bus-off state enabled by default
CANMessage      rxMsg;
CANMessage      txMsg;
int             counter = 0;
volatile bool   msgAvailable = false;

/**
 * @brief   'CAN receive-complete' interrup handler.
 * @note    Called on arrival of new CAN message.
 *          Keep it as short as possible.
 * @param   
 * @retval  
 */
 
void onMsgReceived() {
    msgAvailable = true;
}

/**
 * @brief   Main
 * @note
 * @param 
 * @retval
 */
int main() {
    can.frequency(1000000);                     // set bit rate to 1Mbps
    can.attach(&onMsgReceived, CAN::RxIrq);     // attach 'CAN receive-complete' interrupt handler
    char buff_2[8];
#if defined(BOARD1)
    #if defined(TARGET_STM32F103C8T6)
        led = 0;    // turn LED on
    #else
        led = 1;    // turn LED on
    #endif
    timer.start();
#else
    #if defined(TARGET_STM32F103C8T6)
        led = 1;    // turn LED off
    #else
        led = 0;    // turn LED off
    #endif
#endif
    lcd.cls();
    while(1) {
        wait (0.1);
        lcd.cls();
        lcd.locate(0,3);
        lcd.printf("%d %d %d",char(sensor.temp()),char(pot1*255.0), char(pot2*255.0));
        lcd.locate(0,14);
        lcd.printf("I:%dF:%dl:%d Data:", rxMsg.id, rxMsg.type, rxMsg.len);
        for(int i = 0; i < rxMsg.len; i++)
        {
            lcd.printf("%x", buff_2[i]);
        }
        if(fire) {               // check for timeout
            timer.stop();                       // stop timer
            timer.reset();                      // reset timer (to avaoid repeated send)
            counter++;                          // increment counter
            txMsg.clear();                      // clear Tx message storage
            txMsg.id = TX_ID;                   // set ID
            txMsg << char(sensor.temp());                   // append first data item (make sure that CAN message total data lenght <= 8 bytes!)
            txMsg << char(0x00);
            txMsg << char(pot1*255.0);                // append second data item (make sure that CAN message total data lenght <= 8 bytes!)
            txMsg << char(pot2*255.0); 
            can.write(txMsg);                   // transmit message
            printf("CAN message sent\r\n");
            
           
            
            #if defined(TARGET_STM32F103C8T6)
                led = 1;                        // turn LED off
            #else
                led = 0;                        // turn LED off
            #endif
            while(fire);
        }
        if(msgAvailable) {
            msgAvailable = false;               // reset flag for next use
            can.read(rxMsg);                    // read message into Rx message storage
            printf("CAN message received:\r\n");
            printf("  ID     = %#x\r\n", rxMsg.id);
            printf("  Type   = %d\r\n", rxMsg.type);
            printf("  Format = %d\r\n", rxMsg.format);
            printf("  Length = %d\r\n", rxMsg.len);
            printf("  Data   =");      
            lcd.locate(0,14);
            lcd.printf("I:%dF:%dl:%d", rxMsg.id, rxMsg.type, rxMsg.len);    
            
            for(int i = 0; i < rxMsg.len; i++)
            {
                lcd.printf(" %x", rxMsg.data[i]);
                buff_2[i] =  rxMsg.data[i];
            }   
            r = (rxMsg.data[0]/255.0);
            g = (rxMsg.data[1]/255.0);
            b = (rxMsg.data[2]/255.0);
            if(rxMsg.data[0] == 1) lcd.cls();
            printf("\r\n");            
            if(rxMsg.id == RX_ID) {             // if ID matches
                rxMsg >> counter;               // extract first data item
                rxMsg >> ledReceived;           // extract second data item
                led = ledReceived;              // set LED
                printf("counter = %d\r\n", counter);
                timer.start();
            }
        }
    }
}

