#/usr/bin/env python3

# Remove the SoftDevice part of an nRF51 application, to make it
# transferable over the air

import intelhex

from os.path import exists
from sys import argv, exit


if not argv[2:]:
    print("usage: %s <input hex file> <output name>" % argv[0])
    exit(1)

in_file, out_file = argv[1:3]

# Set start address if autodetection doesn't work
start_address = None

if exists(out_file):
    confirm = input("File %s exists. Replace (y/N)? " % out_file)
    if not confirm.lower().startswith("y"):
        print("Nothing to do.")
        exit(0)


hex_in = intelhex.IntelHex()
hex_in.fromfile(in_file, format='hex')

# Try to guess where application starts: first word contains the
# application's stack base

for app_start in (
        0x16000, # Version 7.1.0 of S110 SoftDevice
        0x18000, #         8.0.0
        0x1c000, #         1.0      S130
        ):

    # Read first word (little-endian)
    try:
        w = hex_in.tobinarray(start=app_start, size=4)
    except intelhex.NotEnoughDataError:
        continue
    word = w[3] << 24 | w[2] << 16 | w[1] << 8 | w[0]

    # Assume stack pointer is at the end of RAM
    if word in (
            0x20004000, # 16K
            0x20008000, # 32K
            ):
        start_address = app_start
        print("Found application at %x" % start_address)
        break

if not start_address:
    print("Application start address not found")
    exit(2)


hex_out = hex_in[start_address:]
hex_out.tofile(out_file, format="hex")
