/*
 * Simple testcase that attempts to collect infos about the SoftDevice, and
 * signal when there is an error. On success, should blink slowly (2s period).
 * On failure, blink every 200ms.
 *
 * In addition, if VERBOSE is defined, show the following messages:
 *
 * For an S130 (1.0.0) Image:
 *     Firmware ID      = 67
 *     Firmware version = s130 (guessed)
 *     S110 flag        = 0
 *    (GattClient scan  = success (0))
 *
 * For an S110 (8.0.0) Image:
 *     Firmware ID      = 64
 *     Firmware version = s110 (guessed)
 *     S110 flag        = 1
 *    (GattClient scan  = not implemented (2))
 *
 * If TEST_BLE is defined, also test GattClient support. With S110, should
 * return BLE_ERROR_NOT_IMPLEMENTED. Otherwise BLE_ERROR_NONE.
 *
 * A PARAM_OUT_OF_RANGE (3) error with S110 most likely means that Central
 * features where not properly disabled.
 * In that case, if S110 flag is 1, it means that the nRF51822 is an old version
 * and does not recognize that flag.
 */
#define VERBOSE
#define TEST_BLE

#include "mbed.h"

DigitalOut led1(LED1);

/* introspection macros */
#include "nrf_sdm.h"

/* An S110 application is supposed to be built with one of those flags */
#if defined(TARGET_MCU_NRF51_16K_S110) || defined(TARGET_MCU_NRF51_32K_S110)
# define S110_MACRO 1
#else
# define S110_MACRO 0
#endif

#ifdef VERBOSE
# define PRINTF printf
#else
# define PRINTF(...)
#endif

#ifdef TEST_BLE
# include "ble/BLE.h"
BLE ble;

void advertisementCallback(const Gap::AdvertisementCallbackParams_t *params)
{
}
#endif

int main(void)
{
    bool error = false;
    uint32_t fwid = SD_FWID_GET(MBR_SIZE);
    /* TODO: find details about the specific version (0x67 is S130 v1.0.0) */
    bool is_s110 = (fwid != 0x67);

    PRINTF("Firmware ID      = %x\r\n", fwid);
    PRINTF("Firmware version = %s (guessed)\r\n", (is_s110 ? "s110" : "s130"));
    PRINTF("S110 flag        = %d\r\n", S110_MACRO);

#ifdef TEST_BLE
    {
        ble.init();

        if (ble.gap().setScanParams(1000, 50) != BLE_ERROR_NONE) {
            PRINTF("Failed to set scan parameters!\r\n");
            return 1;
        }

        /* We expect startScan to return an error when we're using S110 */
        int ret = ble.gap().startScan(advertisementCallback);
        PRINTF("GattClient scan  = %s (%d)\r\n",
                (ret == BLE_ERROR_NONE ?            "success" :
                (ret == BLE_ERROR_NOT_IMPLEMENTED ? "not implemented" :
                                                    "unexpected error")),
                ret);

        if (is_s110 && ret != BLE_ERROR_NOT_IMPLEMENTED)
            error = true;
        if (!is_s110 && ret != BLE_ERROR_NONE)
            error = true;
    }
#endif

    if (S110_MACRO ^ is_s110) /* Check if SD and flags are in sync */
        error = true;

    while (true) {
        /* Blink fast when there is an obvious issue */
        wait_ms(error ? 50 : 1000);
        led1 = !led1;
    }
}
