
#include <mbed.h>
#include "LPS331.h"

#define STATUS_REG         0x27
#define MULTI_BYTE         0x40
#define REF_P_XL           0x08
#define REF_P_L            0x09
#define REF_P_H            0x0A
#define WHO_AM_I           0x0F
#define RES_CONF           0x10
#define CTRL_REG1          0x20
#define PRESS_POUT_XL_REH  0x28
#define PRESS_OUT_L        0x29
#define PRESS_OUT_H        0x2A
#define TEMP_OUT_L         0x2B
#define TEMP_OUT_H         0x2C

#define WRITE              0x00
#define READ               0x80

#define pSensor            0x02
#define tSensor            0x01
#define bothSensors        0x03

lps331::lps331(SPI& _spi, PinName _ncs) : spi(_spi), ncs(_ncs) {
}

uint16_t lps331::read_temperature() {
    // check if updated value is available
    if ( ready(tSensor) )
    {
        select();
        // do multi byte read
        spi.write(READ | MULTI_BYTE | TEMP_OUT_L);
        uint8_t tempL = spi.write(READ);
        uint8_t tempH = spi.write(READ);
        deselect();
        
        uint16_t temperature = (tempH<<8) | tempL;
        // t[C] = 42.5 + temperature /480.0
        //float temp = (float)(42.5 + ((short)temperature*1.0/480.0));
        return temperature;
    }
    return 0;
}

uint32_t lps331::read_pressure() {
    // check if updated value is available
    if ( ready(pSensor) )
    {
        select();
        // do multi byte read
        spi.write(READ | MULTI_BYTE | PRESS_POUT_XL_REH);
        uint8_t pressureLX = spi.write(READ);
        uint8_t pressureL = spi.write(READ);
        uint8_t pressureH = spi.write(READ);
        deselect();
        
        uint32_t pressure = (pressureH<<16) | (pressureL<<8) | pressureLX;
        // P[mBar] = pressure/4096
        // float P = pressure*1.0/4096.0;
        return pressure;
    }
    return 0;
}

float lps331::get_Pressure_mbar()
{
    uint32_t airP = read_pressure();
    float pressure = airP*1.0/4096.0;
    return pressure;
}

float lps331::get_Temp_C()
{
    short airT = read_temperature();
    float temp = (float)(42.5 + (airT*1.0/480.0));
    return temp;
}

float lps331::get_Temp_F()
{
    short airT = read_temperature();
    float temp = (float)(108.5 + (airT*1.8/480.0));
    return temp;
}

void lps331::select() {
    //Set CS low to start transmission (interrupts conversion)
    ncs = 0;
}

void lps331::deselect() {
    //Set CS high to stop transmission (restarts conversion)
    ncs = 1;
}

bool lps331::initialize(int setType) {
    deselect();
    spi.format(8,0);
    spi.frequency(400000);  
    
    // set chip to update pressure and temp at 25Hz
    writeRegister(CTRL_REG1, 0xE0);
    wait_ms(1);
    
    // check board is responding
    int whoami = readRegister(WHO_AM_I);
    if (whoami == 0xBB)  // addresss of chip
        return true;
    return false;
}

int lps331::ready(uint8_t sensor) {
    uint8_t status = readRegister(STATUS_REG);
    if( (status & sensor))
        return 1;
    return 0;
}

uint8_t lps331::readRegister(uint8_t address){
    select();
    spi.write(READ | address);
    uint8_t val = spi.write(READ);
    deselect();
    return val;
    }

void lps331::writeRegister(uint8_t address, uint8_t value){
     select();
    spi.write(WRITE | address);
    spi.write(value);
    deselect();
}