#include "mbed.h"
#include "mDot.h"
#include "rtos.h"
#include "ChannelPlan.h"
#include "ISL29011.h"
#include "dot_util.h"
#include "RadioEvent.h"

static uint8_t network_address[] = { 0x04, 0x05, 0x03, 0x04 };
static uint8_t network_session_key[] = { 0x04, 0x05, 0x03, 0x04, 0x04, 0x05, 0x03, 0x04, 0x04, 0x05, 0x03, 0x04, 0x04, 0x05, 0x03, 0x04 };
static uint8_t data_session_key[] = { 0x04, 0x05, 0x03, 0x04, 0x04, 0x05, 0x03, 0x04, 0x04, 0x05, 0x03, 0x04, 0x04, 0x05, 0x03, 0x04 };

mDot* dot = NULL;
lora::ChannelPlan* plan = NULL;

static Serial pc(USBTX, USBRX);

I2C i2c(I2C_SDA, I2C_SCL);
ISL29011 lux(i2c);

int main() {
    
    // Custom event handler for automatically displaying RX data
    RadioEvent events;
    uint32_t tx_frequency;
    uint8_t tx_datarate;
    uint8_t tx_power;
    
    pc.baud(115200);
    
    plan = new lora::ChannelPlan_US915();
    dot = mDot::getInstance(plan);
    assert(dot);
    
    logInfo("defaulting Dot configuration");
    dot->resetConfig();
    
    // make sure library logging is turned on
    dot->setLogLevel(mts::MTSLog::INFO_LEVEL);

    // attach the custom events handler
    dot->setEvents(&events);

    // update configuration if necessary
    if (dot->getJoinMode() != mDot::PEER_TO_PEER) {
        logInfo("changing network join mode to PEER_TO_PEER");
        if (dot->setJoinMode(mDot::PEER_TO_PEER) != mDot::MDOT_OK) {
            logError("failed to set network join mode to PEER_TO_PEER");
        }
    }
    
    tx_frequency = 915500000;
    tx_datarate = lora::DR_13;
    // 915 bands have no duty cycle restrictions, set tx power to max
    tx_power = 20;
    
    // in PEER_TO_PEER mode there is no join request/response transaction
    // as long as both Dots are configured correctly, they should be able to communicate
    update_peer_to_peer_config(network_address, network_session_key, data_session_key, tx_frequency, tx_datarate, tx_power);

    // save changes to configuration
    logInfo("saving configuration");
    if (!dot->saveConfig()) {
        logError("failed to save configuration");
    }

    // display configuration
    display_config();

    lux.setMode(ISL29011::ALS_CONT);
    lux.setResolution(ISL29011::ADC_16BIT);
    lux.setRange(ISL29011::RNG_64000);
    
      while (true) {
        uint16_t light;
        std::vector<uint8_t> tx_data;

        // join network if not joined
        if (!dot->getNetworkJoinStatus()) {
            join_network();
        }
        
    light = lux.getData();
    tx_data.push_back((light >> 8) & 0xFF);
    tx_data.push_back(light & 0xFF);
    logInfo("light: %lu [0x%04X]", light, light);
    send_data(tx_data);  
    
    // the Dot can't sleep in PEER_TO_PEER mode
        // it must be waiting for data from the other Dot
        // send data every 5 seconds
        logInfo("waiting for 5s");
        wait(5);  
    }
return 0;
}