#ifndef WEAPON_H
#define WEAPON_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Spawn.h"
#include "GUI.h"

#define Rows 48
#define Cols 84

//position of the bomb target
struct Target {
    int xx;
    int yy;    
};

//to store the weapon level
struct Variables {
    int damage;
    int range;
    int cooldown;
    int capacity;
    int regenRate;

};

/** Weapon Class
@brief Library that contains the weapons for the game 
@author JianWei Lee
@date April 2017
*/
class Weapon
{
public:
    Weapon();
    ~Weapon();
    
    void init();
    
    /**Draw player
    *
    *@brief Draw player on the screen at (43,46) \n
    *@brief Draw the position of the weapon depending on the joystick direction
    */
    void drawPlayer(Gamepad &pad,N5110 &lcd);
    
    /**Math calculation 
    *
    *@brief It gets the gradient from gamepad and calculate y-intercept of the line using the player's position \n
    *@brief The math is required for laser and cannon.
    */
    void weaponMath(Gamepad &pad);
    
    /**Laser
    *
    *@brief The main code for laser \n
    *@brief Button A to activate laser \n
    *@brief The screen is divided into 4 parts for better line resolution and gradient changes in these different region \n
    *@brief Laser can only deal damage to spawn A, C and DE. \n
    *@brief Laser path will be blocked by spawn B. 
    *@param Arr2[][Rows] - deal damage to the spawn
    *@param cArr2[][Rows] - to differentiate the spawns.
    */
    void laser_Main(int Arr2[][Rows], char cArr2[][Rows], N5110 &lcd);
    
    /*Bomb
    *
    *@brief The main code for bomb \n
    *@brief Button L to bring up the bomb target, button X to detonate bomb \n
    *@brief Bomb deals damage to an area - area size depends on the bomb radius \n
    *@brief A circle will be drawn on the area bombed. \n
    *@brief Bomb can only deal damage to spawn A, C and DE.
    *@param Arr2[][Rows] - deal damage to the array
    */
    void bomb_main(int Arr2[][Rows], Gamepad &pad, N5110 &lcd);
    
    
   
    /**Shield
    *
    *@brief The main code for shield \n
    *@brief Button B to activate/deactivate shield. \n
    *@brief Shield covers the player from fast moving spawn (spawn B and D&E) at y = 45 \n
    *@brief The shield will be deactivated when depleted and will only recharge when the shield is deactivated
    *@param Arr2[][Rows] - deal damage to the spawn
    *@param cArr2[][Rows] - to differentiate the spawns.
    */
    void shield_main(int Arr2[][Rows], char cArr2[][Rows], Gamepad &pad, N5110 &lcd);
    
    
    /**Initialise drone
    *
    *@brief Drone on left side pixel 0 to 42 \n
    *@brief Drone on right side pixel 42 to 83 
    *@param x - position x of drone
    *@param y - position y of drone
    *@param a - left border of the drone range
    *@param b - right border of the drone range
    *@param damage - damage of the drone
    *@param range - range of the drone
    */
    void droneInit(int x, int y, int a, int b, int damage, int range);
    
    /*Drone
    *
    *@brief The drone will automatically deal damage to spawn A, C and D&E with a given range \n
    *@brief Drone will prioritize at attacking spawn D&E when it enters the drone range 
    */
    void drone_main(int Arr2[][Rows], char cArr2[][Rows], N5110 &lcd);
    
    
    /**Cannon
    *
    *@brief Cannon only can be fired when laser is switched on (button Y) \n
    *@brief Cannon can delete spawn A, B and C instantly on contact \n
    *@brief It's damage indicate how many spawns it can collide with  \n
    *@brief The cannon has a fixed radius and travel at the same speed at spawn B  \n
    *@brief Cannon ammo is not unlimited and will recharge whenever the ammo is not full 
    */
    void cannon_main(int Arr2[][Rows], char cArr2[][Rows], N5110 &lcd);
    
    /**Cannon meter
    *
    *@brief An indicator for the remaining cannon ammo \n
    *@brief Cannon ammo size can be upgraded
    */
    void cannon_Meter(N5110 &lcd);
    
    /**Cannon recharge
    *
    *@brief Recharge cannon ammo whenever the cannon ammo is not full \n
    *@brief Cannon recharge rate can be upgraded 
    */
    void cannon_recharge();
    
    /**Set Variables
    *
    *@brief This code will be executed after weapon upgrades \n
    *@brief Set the variables using numbers from GUI's struct 
    *@param up - weapon's number 
    *@param up1 - variable's number
    */
    void setVar(int up, int up1);
    
    /**Clear weapon variables
    *
    *@brief To be used only when the player give up on the game. \n
    *@brief All weapon variables are set to 0
    */
    void reset_WeaponVariables();
    
    //these variable can be upgraded as the game progress
    int _bombRange;
    int _laserDamage;
    int _bombDamage;
    int _bombCooldown;
    int _shieldCapacity;
    int _shieldRegenRate;
    int _droneDamage;
    int _droneRange;
    int _cannonDamage;  
    int _cannonCapacity;
    int _cannonRegenRate;
    
    //these are used in game engine 
    int cannonFiring_flag;
    int _cannon;
    
private:
    //structs 
    Variables laser;
    Variables bomb;
    Variables drone;
    Variables shield;
    Variables cannon;
    
    //Laser
    int c;
    int y;
    int x;
    float m;
    int _x;
    int _y;
    int lineBreak;
    //screen divided into 4 parts
    void laser1and4(int Arr2[][Rows], char cArr2[][Rows], N5110 &lcd);
    void laser2(int Arr2[][Rows], char cArr2[][Rows], N5110 &lcd);
    void laser3(int Arr2[][Rows], char cArr2[][Rows], N5110 &lcd);
    //look for spawn A, B, C and D&E
    void laser_detectSpawn(int i, int j, int Arr2[][Rows], N5110 &lcd); 
    //to give C a larger hitbox
    void laser_detectSpawnC(int i, int j, int Arr2[][Rows], char cArr2[][Rows], N5110 &lcd); 
    
    //drone
    int breakout;
    int d_x;
    int d_y;
    int d_side; //which side the drone is at. 0 for left side. 42 for right side.
    int d_scan;
    int d_foundDE;
    //draw a line for laser
    void droneLaser(int x, int y, int Arr2[][Rows], char cArr2[][Rows], N5110 &lcd);
    //drone scan for spawn A, B and D&E
    void droneScanForAB(int Arr2[][Rows], char cArr2[][Rows], N5110 &lcd);
    void droneScanForDE(int Arr2[][Rows], char cArr2[][Rows], N5110 &lcd);
    
    //cannon
    float cannonDirection;
    int cannon_y;
    int cannon_x;
    int cannon_c;
    int cannon_Damage;
    //screen divided into 4 parts
    void cannon1and4(int Arr2[][Rows], char cArr2[][Rows], N5110 &lcd); 
    void cannon2(int Arr2[][Rows], char cArr2[][Rows], N5110 &lcd);
    void cannon3(int Arr2[][Rows], char cArr2[][Rows], N5110 &lcd);
    //draw cannon ball
    void draw_CannonBall(int x, int y, N5110 &lcd);
    //large cannon hit box 
    void cannon_HitBox(int i, int j, int Arr2[][Rows], char cArr2[][Rows]);
    
    
    
    
    //bomb
    int _bomb;
    int bombTarget_flag;
    //initialise bomb
    void bombInit();
    //draw bomb target on screen
    void drawBombTarget(Gamepad &pad, N5110 &lcd);
    //the function to toggle bomb target flag
    void checkBombTarget(Gamepad &pad);
    //http://stackoverflow.com/questions/17163636/filled-circle-in-matrix2d-array
    //draw a circle on screen using pythagoras theorem
    //and deal damage to the area
    void drawCircle(int a, int b, int r, int c2, int x2, int y2, int Arr2[][Rows], N5110 &lcd);
    void spawnDamageByBomb(int Arr2[][Rows], N5110 &lcd);
    //struct to store bomb target's coordinate
    Target targetCoord(Gamepad &pad); 
    
    //shield
    int _shield;
    //Scan for spawn B and D&E and deletes them at y = 45.
    void shield_Rule(int Arr2[][Rows], char cArr2[][Rows], N5110 &lcd);
    //draw shield
    void drawEnergyShield(N5110 &lcd);
    void shieldMeter(N5110 &lcd);
    
};

#endif
    