#ifndef SPAWN_H
#define SPAWN_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Weapon.h"

//Y rows and X columns
#define Rows 48
#define Cols 84

//store the stats of spawn A, B, C and D&E.
struct Stats {
    int health;
    int spawnRate;
};

/** Spawn Class
@brief Library that contain all the enemy in the game 
@author JianWei Lee
@date April 2017
*/
class Spawn
{
public:
    Spawn();
    ~Spawn();
    
    /**Initialise Spawn
    *
    *@brief The health and spawn rate of the spawn depends on the number of level \n
    *@brief Randomise the seed to make the spawning pattern different for every level
    *@param w - the number of level
    */
    void init(int w);
    
    /**Summon Spawn
    *
    *@brief Spawn A and C takes turn to be summoned to prevent collision (summon spawn A this row and spawn C the next row) \n
    *@brief Spawn B and DE will be summoned randomly depends solely on the spawn rate
    *@param Arr[][Rows] - int Array to store spawn health
    *@param cArr2[][Rows] - char Array to store the spawn character
    */
    void summon_spawn(int Arr[][Rows], char cArr[][Rows]);
    
    /** Movement of Spawn A, B and C
    *
    *@brief Moving behavior of spawn A, B and C is just moving down a row
    *@param Arr[][Rows] - the int Array in sync with the char Array to be scanned
    *@param Arr2[][Rows] - int Array to be updated
    *@param cArr[][Rows] - char Array to be scanned
    *@param cArr2[][Rows] - char Array to be updated
    */
    void moveSpawnABC(int Arr[][Rows], int Arr2[][Rows], char cArr[][Rows], char cArr2[][Rows]);
    
    /**Movement of Spawn B
    *
    *@brief Move spawn B down a row 
    *@param Arr[][Rows] - the int Array in sync with the char Array to be scanned
    *@param Arr2[][Rows] - int Array to be updated
    *@param cArr[][Rows] - char Array to be scanned
    *@param cArr2[][Rows] - char Array to be updated
    */
    void moveSpawnB(int Arr[][Rows], int Arr2[][Rows], char cArr[][Rows], char cArr2[][Rows]);
    
    /**Movement of Spawn D&E
    *
    *@brief Spawn D & E moves diagonally \n
    *@brief Spawn D - move left & down \n
    *@brief Spawn E - move right & down \n
    *@brief It moves in the other direction when hit 'L' or 'R' of spawnC and walls. \n
    *@brief It jumps over Spawn A and Spawn B
    *@param Arr[][Rows] - the int Array in sync with the char Array to be scanned
    *@param Arr2[][Rows] - int Array to be updated
    *@param cArr[][Rows] - char Array to be scanned
    *@param cArr2[][Rows] - char Array to be updated
    */
    void moveSpawnDE(int Arr[][Rows], int Arr2[][Rows], char cArr[][Rows], char cArr2[][Rows]);
    
    /**Update spawn
    *
    *@brief This function to update spawn from secondary array to main array after taking damage & after moving spawn \n
    *@brief It also prints the spawn on LCD 
    *@param Arr[][Rows] - int Array to be updated
    *@param Arr2[][Rows] - the int Array in sync with the char Array to be scanned
    *@param cArr[][Rows] - char Array to be updated
    *@param cArr2[][Rows] - char Array to be scanned 
    */
    void updateSpawn(int Arr[][Rows], int Arr2[][Rows], char cArr[][Rows], char cArr2[][Rows], N5110 &lcd);
    
    /*Delete spawn
    *
    *@brief Spawn that has 0 health will be deleted (cleaning tool) \n
    *http://stackoverflow.com/questions/10289197/how-to-empty-a-2d-char-array-in-c
    *@param x - the x co-ordinate of the spawn (0 to 83)
    *@param y - the y co-ordinate of the spawn (0 to 47)
    *@param Arr[][Rows] - the int array that will be scanned
    *@param cArr[][Rows] - the char array that will be cleaned/deleted if Arr[x][y] == 0
    */
    void deleteChar(int x, int y, int Arr[][Rows], char cArr[][Rows]);
     
private:
    
    int stopSpawnA;
    
    //create structs to store the stats of the spawns
    Stats A;
    Stats B;
    Stats C;
    Stats DE;
    
    //Summon spawns
    void spawnA(int Arr[][Rows],char cArr[][Rows]); //normal spawn
    void spawnB(int x, int Arr[][Rows],char cArr[][Rows]); //fast moving spawn
    void spawnC(int Arr[][Rows],char cArr[][Rows]); //tanky spawn
    void spawnDE(int x, int Arr[][Rows],char cArr[][Rows]); //zigg-zag spawn (even faster than spawn B)
    
    //Movement of spawn D&E.
    void movementD(int x, int y, int Arr[][Rows], int Arr2[][Rows], char cArr[][Rows], char cArr2[][Rows]);
    void movementE(int x, int y, int Arr[][Rows], int Arr2[][Rows], char cArr[][Rows], char cArr2[][Rows]);
    
};

#endif 