#include "mbed.h"
#include "ble/BLE.h"
#include "UltrasonicService.h"
#include "hcsr04.h"

#define ULTRASONIC_UPDATE_INTERVAL          1.0f        // in seconds

#define LED_ON          0
#define LED_OFF         1

static DigitalOut alivenessLED(LED1, LED_OFF); // green
static HCSR04 usensor(D0, D1);

const static char     DEVICE_NAME[] = "HPE_ULTRASONIC";
static const uint16_t uuid16_list[] = { 
    UltrasonicService::SERVICE_UUID
};

static UltrasonicService *uServicePtr;

static Ticker ledTicker;
static Ticker uTicker;
static Timeout cbTimeout;

void readDistanceCallback(void)
{
    uint32_t dist = usensor.get_dist_cm();

    uServicePtr->updateState(dist);
    // should add a stop function? usensor.stop();
}
    

void readDistance(void)
{
    usensor.start();
    
    // need to like have it on for for 0.5s
    cbTimeout.attach(&readDistanceCallback, 0.5f);
}

void periodicCallback(void)
{
    alivenessLED = !alivenessLED; /* Do blinky on LED1 to indicate system aliveness. */
}

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    printf("Disconnected over BLE!\r\n");
    
    BLE::Instance().gap().startAdvertising();
    
    ledTicker.attach(periodicCallback, 1);
    uTicker.detach();
}

void connectionCallback(const Gap::ConnectionCallbackParams_t * params)
{
    printf("Connected over BLE!\r\n");
    
    // blink a bit faster when connected
    ledTicker.attach(periodicCallback, 0.5);
    
    // only do ultrasonic readings whenever we're connected over BLE
    uTicker.attach(readDistance, ULTRASONIC_UPDATE_INTERVAL);
}

void onBleInitError(BLE &ble, ble_error_t error)
{
    // blink fast when we encountered an error
    ledTicker.attach(periodicCallback, 0.2);
}
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }
 
    ble.gap().onDisconnection(disconnectionCallback);
    ble.gap().onConnection(connectionCallback);

    // Begin - If you add a new service, add it here!
    uServicePtr = new UltrasonicService(ble);
    // End - If you add a new service, add it here!

    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(1000);
    ble.gap().startAdvertising();
}

int main(void)
{
    // Blink the green LED!
    ledTicker.attach(periodicCallback, 1);
    
    BLE &ble = BLE::Instance();
    ble.init(bleInitComplete);

    /* SpinWait for initialization to complete. This is necessary because the
     * BLE object is used in the main loop below. */
    while (ble.hasInitialized()  == false) { /* spin loop */ }

    while (true) {
        ble.waitForEvent();
    }
}
