/*-
 * BSD 2-Clause License
 *
 * Copyright (c) 2012-2018, Jan Breuer
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * * Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * @file   scpi-def.c
 * @date   Thu Nov 15 10:58:45 UTC 2012
 *
 * @brief  SCPI parser test
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "scpi/scpi.h"
#include "scpi-def.h"

#include "MicroBit.h"

extern MicroBit uBit;
MicroBitImage imgPass("0, 0, 255, 0, 0\n0, 0, 255, 0, 0\n255, 255, 255, 255, 255\n0, 0, 255, 0, 0\n0, 0, 255, 0, 0\n"); // plus
MicroBitImage imgFail("255, 0, 0, 0, 255\n0, 255, 0, 255, 0\n0, 0, 255, 0, 0\n0, 255, 0, 255, 0\n255, 0, 0, 0, 255\n"); // X


enum scpi_state_t {
    SCPI_STATE_PASS,
    SCPI_STATE_FAIL,
    SCPI_STATE_CLEAR
};

const scpi_choice_def_t scpi_state_def[] = {
    {/* name */ "PASs",      /* type */ SCPI_STATE_PASS  },
    {/* name */ "FAIl",      /* type */ SCPI_STATE_FAIL  },
    {/* name */ "CLEar",    /* type */ SCPI_STATE_CLEAR  },
    SCPI_CHOICE_LIST_END,
};


/**
 * Reimplement IEEE488.2 *TST?
 *
 * Result should be 0 if everything is ok
 * Result should be 1 if something goes wrong
 *
 * Return SCPI_RES_OK
 */
static scpi_result_t My_CoreTstQ(scpi_t * context) {

    SCPI_ResultInt32(context, 0);

    return SCPI_RES_OK;
}

static scpi_result_t setState(scpi_t * context) {

    int32_t param1;
    scpi_bool_t result;
    scpi_result_t retval;

    result = SCPI_ParamChoice( context, scpi_state_def, &param1, TRUE );
    if ( false == result )
    {
        return SCPI_RES_ERR;
    }
    else
    {
        switch (param1) {
        case SCPI_STATE_PASS:
            uBit.io.P1.setDigitalValue(0);
            uBit.io.P0.setDigitalValue(1);
            uBit.display.clear();
            uBit.display.print(imgPass);
            retval = SCPI_RES_OK;
            break;
        case SCPI_STATE_FAIL:
            uBit.io.P0.setDigitalValue(0);
            uBit.io.P1.setDigitalValue(1);
            uBit.display.clear();
            uBit.display.print(imgFail);
            retval = SCPI_RES_OK;
            break;
        case SCPI_STATE_CLEAR:
            uBit.io.P0.setDigitalValue(0);
            uBit.io.P1.setDigitalValue(0);
            uBit.display.clear();
            retval = SCPI_RES_OK;
            break;
        default:
            SCPI_ErrorPush(context, SCPI_ERROR_ILLEGAL_PARAMETER_VALUE);
            retval = SCPI_RES_ERR;
       }
    }

    return retval;
}

const scpi_command_t scpi_commands[] = {
    /* IEEE Mandated Commands (SCPI std V1999.0 4.1.1) */
     /* IEEE Mandated Commands (SCPI std V1999.0 4.1.1) */
    { .pattern = "*CLS", .callback = SCPI_CoreCls,},
    { .pattern = "*ESE", .callback = SCPI_CoreEse,},
    { .pattern = "*ESE?", .callback = SCPI_CoreEseQ,},
    { .pattern = "*ESR?", .callback = SCPI_CoreEsrQ,},
    { .pattern = "*IDN?", .callback = SCPI_CoreIdnQ,},
    { .pattern = "*OPC", .callback = SCPI_CoreOpc,},
    { .pattern = "*OPC?", .callback = SCPI_CoreOpcQ,},
    { .pattern = "*RST", .callback = SCPI_CoreRst,},
    { .pattern = "*SRE", .callback = SCPI_CoreSre,},
    { .pattern = "*SRE?", .callback = SCPI_CoreSreQ,},
    { .pattern = "*STB?", .callback = SCPI_CoreStbQ,},
    { .pattern = "*TST?", .callback = My_CoreTstQ,},
    { .pattern = "*WAI", .callback = SCPI_CoreWai,},

    /* Required SCPI commands (SCPI std V1999.0 4.2.1) */
    {.pattern = "SYSTem:ERRor[:NEXT]?", .callback = SCPI_SystemErrorNextQ,},
    {.pattern = "SYSTem:ERRor:COUNt?", .callback = SCPI_SystemErrorCountQ,},
    {.pattern = "SYSTem:VERSion?", .callback = SCPI_SystemVersionQ,},

    {.pattern = "[:INDicator]:STAte", .callback = setState,},


    SCPI_CMD_LIST_END
};

scpi_interface_t scpi_interface = {
    /*.error = */ SCPI_Error,
    /*.write = */ SCPI_Write,
    /*.control = */ SCPI_Control,
    /*.flush = */ SCPI_Flush,
    /*.reset = */ SCPI_Reset,
};

char scpi_input_buffer[SCPI_INPUT_BUFFER_LENGTH];
scpi_error_t scpi_error_queue_data[SCPI_ERROR_QUEUE_SIZE];

scpi_t scpi_context;
