#ifndef ICMP_H
#define ICMP_H

#include "net.h"

/**
  \file icmp.h
  \brief ICMP frame header
  
  This file contains the memory map and associated functions for ICMP packet
  creation and deconstruction. 
*/

#define ICMP_ECHO_REPLY   0x00
#define ICMP_ECHO_REQUEST 0x08
#define IPPROTO_ICMP      0x01

/// ICMP packet memory map
typedef struct {
    u8 type;        ///< type of ICMP message
    u8 code;        ///< code number associated with certain message types
    u16 checksum; 
    u16 id;         ///< ID value, returned in ECHO REPLY
    u16 sequence;   ///< Sequence value to be returned with ECHO REPLY
    u8 data[];      ///< Data memory map
} ICMP_Packet;

/// Convert from wire to host or host to wire endianness
inline void fix_endian_icmp(ICMP_Packet *segment)
{
  fix_endian_u16(&segment->checksum);
  fix_endian_u16(&segment->id);
  fix_endian_u16(&segment->sequence);
}

/// Print the ICMP packet
inline void print_icmp(ICMP_Packet *segment)
{
  printf("ICMP Packet:");
  printf("  Type:      0x%02X \n", segment->type);
  printf("  Code:      0x%02X \n", segment->code);
  printf("  Checksum:  0x%04X \n", segment->checksum);
  printf("  ID:        0x%04X \n", segment->id);
  printf("  Sequence:  0x%04X \n", segment->sequence);
}

#endif