#ifndef ETHERNET_H
#define ETHERNET_H

#include "net.h"

/**
  \file ethernet.h
  \brief Ethernet frame header
  
  This file contains the memory map and associated functions for Ethernet frame header
  creation and deconstruction. 
*/

/// Ethernet MAC address memory map
typedef struct {
  unsigned char octet[6]; ///< Individual octsts of the MAC address
} Ethernet_MAC;

/// Ethernet II Frame Header Memory map
typedef struct {
  /// Destination MAC address (6 octets)
  Ethernet_MAC destination;
  /// Source MAC address (6 octets)
  Ethernet_MAC source;
  // (optional) VLAN Tag (unsupported)
  /// Ethernet type or length (only <0x600 or 0x0800 IPv4 supported)
  u16 ethertype;
  /// Payload (used for memory mapping; has zero size)
  unsigned char payload[];
} Ethernet_FrameHeader;

/// Convert from wire to host or host to wire endian-ness
inline void fix_endian_ethernet(Ethernet_FrameHeader *header)
{
  fix_endian_u16(&header->ethertype);
}

/// Print out an ethernet packet
inline void print_ethernet(Ethernet_FrameHeader *frame)
{
  printf("Ethernet frame: \n");
  u8 *src = frame->source.octet;
  u8 *dst = frame->destination.octet;
  printf("  Source:    MAC - %02X:%02X:%02X:%02X:%02X:%02X \n", src[6], src[7], src[8], src[9], src[10], src[11]);
  printf("  Dest:      MAC - %02X:%02X:%02X:%02X:%02X:%02X \n", dst[0], dst[1], dst[2], dst[3], dst[4], dst[5]);
  printf("  Ethertype: 0x%04X \n", frame->ethertype);
}

#endif