#ifndef ARP_H
#define ARP_H

#include "net.h"

/**
  \file
  \brief ARP packet
  
  This file contains the memory map and associated functions for ARP packet
  creation and deconstruction. 
*/

#define ETHERTYPE_ARP 0x0806

/// ARP Packet memory map
typedef struct {
  u16 hardware_type; ///< 0x0001 for ethernet
  u16 protocol_type; ///< 0x0800 for IPv4
  u8  hardware_length; ///< Bytes.  Ethernet is 6
  u8  protocol_length; ///< Bytes.  IPv4 is 4
  u16 operation; ///< Operation.  1 for request, 2 for reply or announce
  // The following are only valid for IPv4 over Ethernet
  u8  sender_hardware_address[6]; ///< Generator of the request or reply
  u8  sender_protocol_address[4]; ///< All zeroes for an ARP probe
  u8  target_hardware_address[6]; ///< Announce - same as SHA
  u8  target_protocol_address[4]; ///< Announce - Same as TPA
} ARP_Packet;

/// Convert from wire to host or host to wire endianness
inline void fix_endian_arp(ARP_Packet *packet)
{
  fix_endian_u16(&packet->hardware_type);
  fix_endian_u16(&packet->protocol_type);
  fix_endian_u16(&packet->operation);
}

/// Print the ARP packet
inline void print_arp(ARP_Packet *packet)
{
    printf("ARP Packet: \n");
    printf("  Hardware:  0x%04X \n", packet->hardware_type);
    printf("  Protocol:  0x%04X \n", packet->protocol_type);
    printf("  Type:      %d \n", packet->operation);
    if (packet->hardware_type != 0x0001 || packet->protocol_type != 0x0800) return;
    
    u8 *bytes;
    bytes = packet->sender_hardware_address;
    printf("  Source:    MAC - %02X:%02X:%02X:%02X:%02X:%02X \n", bytes[0],bytes[1],bytes[2],bytes[3],bytes[4],bytes[5]);
    bytes = packet->target_hardware_address;
    printf("  Target:    MAC - %02X:%02X:%02X:%02X:%02X:%02X \n", bytes[0],bytes[1],bytes[2],bytes[3],bytes[4],bytes[5]);
    bytes = packet->sender_protocol_address;
    printf("  Source:    IP  - %03d.%03d.%03d.%03d \n", bytes[0], bytes[1], bytes[2], bytes[3]);
    bytes = packet->target_protocol_address;
    printf("  Target:    IP  - %03d.%03d.%03d.%03d \n", bytes[0], bytes[1], bytes[2], bytes[3]);
}

#endif