/*
 * ADC Sensor Demo
 *
 * Analog inputs are read and pushed to the LoRa gateway.
 *
 * Analog sensors should be wired to mDot pins 16 and 17.
 *
 */

#include "mbed.h"
#include "mDot.h"
#include <string>
#include <vector>

/*
 * LoRa settings
 *
 * For Conduit AEP settings can be found at:
 *
 *   https://<ip-address>/lora_network.html
 *
 * For Conduit mlinux:
 *
 *   /var/config/lora/lora-network-server.conf
 *
 */
static std::string config_network_name = "testtest";
static std::string config_network_pass = "memememe";
static uint8_t config_frequency_sub_band = 5;

enum {
    MSG_TYPE_ADC0,
    MSG_TYPE_ADC1,
};

AnalogIn adc0(PA_1); // PIN16 on mDot
AnalogIn adc1(PA_4); // PIN17 on mDot

Serial debug(USBTX, USBRX);

int main() {
    int32_t rc;

    debug.baud(115200);

    mDot *mdot = mDot::getInstance();

    mdot->resetConfig();

    printf("mDot version: %s\r\n", mdot->getId().c_str());

    // Activity Status on UDK2
    mdot->setActivityLedPin(LED1);
    mdot->setActivityLedEnable(true);

    if ((rc = mdot->setFrequencySubBand(config_frequency_sub_band)) != mDot::MDOT_OK) {
        printf("failed to set frequency sub-band: %d\r\n", rc);
        return 1;
    }

    if ((rc = mdot->setNetworkName(config_network_name)) != mDot::MDOT_OK) {
        printf("failed to set network name: %d\r\n", rc);
        return 1;
    }

    if ((rc = mdot->setNetworkPassphrase(config_network_pass)) != mDot::MDOT_OK) {
        printf("failed to set network pass phrase: %d\r\n", rc);
        return 1;
    }

    while ((rc = mdot->joinNetwork()) != mDot::MDOT_OK) {
        printf("failed to join network: %d\r\n", rc);
        if (mdot->getFrequencyBand() == mDot::FB_868){
            rc = mdot->getNextTxMs();
        } else {
            rc = 0;
        }

        wait(1);
    }

    while (true) {
        std::vector<uint8_t> lora_pl;

        // Read analog inputs
        uint16_t adc0_value = adc0.read_u16();
        uint16_t adc1_value = adc1.read_u16();

        // Pack ADC0 TLV
        lora_pl.clear();
        lora_pl.push_back(MSG_TYPE_ADC0);
        lora_pl.push_back(2);
        lora_pl.push_back(((char *) &adc0_value)[1]);
        lora_pl.push_back(((char *) &adc0_value)[0]);

        // Pack ADC1 TLV
        lora_pl.push_back(MSG_TYPE_ADC1);
        lora_pl.push_back(2);
        lora_pl.push_back(((char *) &adc1_value)[1]);
        lora_pl.push_back(((char *) &adc1_value)[0]);

        // Send message
        while ((rc = mdot->send(lora_pl)) != mDot::MDOT_OK) {
            printf("failed to send message: %d\r\n", rc);
            wait(1);
        }

        wait(5);
    }
}
