/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef USB_HID_H
#define USB_HID_H

/* These headers are included for child class. */
#include "USBEndpoints.h"
#include "USBDescriptor.h"
#include "USBDevice_Types.h"

#include "USBHID_Types.h"
#include "USBDevice.h"

/**
 * USBHID example
 * @code
 * 
 * //We declare a USBHID device. Input, Output, and Feature reports have a length of 8 bytes
 * USBHID hid(8, 8, 8,0x1234,0x0006, 0x0001,true);
 * HID_REPORT tempReport;
 
 * void MycallbackSetReport(HID_REPORT *report){
 *     printf("Report: ");
 *             for(int i = 0; i < report->length; i++) {
 *                 printf("%x ", report->data[i]);
 *             }
 *     printf("\r\n");
 * };
 * 
 * bool SetTempReport(HID_REPORT *report){
 *         for (int i = 0; i < report->length; i++) {
 *             report->data[i] = rand() & 0xff;
 *         }
 *     return true;
 * };
 *  
 * int main(void) {
 *     printf("Starting==>\r\n");
 *     hid.callbackSetOutputReport=&MycallbackSetReport;    
 *     tempReport.length=8;

 *     while (1) {
 *        SetTempReport(&tempReport);
 *        hid.FillInputReport(&tempReport);        
 *        wait(2.2);
 *     }
 * }
 * 
 * @endcode
 */

class USBHID: public USBDevice {
public:

    /**
    * Constructor
    *
    * @param output_report_length Maximum length of a sent report (up to 64 bytes) (default: 64 bytes)
    * @param input_report_length Maximum length of a received report (up to 64 bytes) (default: 64 bytes)
    * @param report_report_length Maximum length of a received report (up to 64 bytes) (default: 64 bytes)
    * @param vendor_id Your vendor_id
    * @param product_id Your product_id
    * @param product_release Your preoduct_release
    * @param connect Connect the device
    */
    USBHID(uint8_t output_report_length , uint8_t input_report_length , uint8_t feature_report_length , uint16_t vendor_id , uint16_t product_id , uint16_t product_release , bool connect );
    /**
    * Constructor
    * default constructor to maintain backwards compatibility. 
    *
    * @param output_report_length Maximum length of a sent report (up to 64 bytes) (default: 64 bytes)
    * @param input_report_length Maximum length of a received report (up to 64 bytes) (default: 64 bytes)
    * @param report_report_length Maximum length of a received report (up to 64 bytes) (default: 64 bytes)
    * @param vendor_id Your vendor_id
    * @param product_id Your product_id
    * @param product_release Your preoduct_release
    * @param connect Connect the device
    */
    USBHID(uint8_t output_report_length = 64, uint8_t input_report_length = 64, uint16_t vendor_id = 0x1234, uint16_t product_id = 0x0006, uint16_t product_release = 0x0001, bool connect = true);


    /**
    * Send a Report. warning: blocking
    * legacy method retained for backward compatability
    *
    * @param report Report which will be sent (a report is defined by all data and the length)
    * @returns true if successful
    */
    bool send(HID_REPORT *report);
    
    
    /**
    * Send a Report. warning: non blocking
    * legacy method retained for backward compatability
    *
    * @param report Report which will be sent (a report is defined by all data and the length)
    * @returns true if successful
    */
    bool sendNB(HID_REPORT *report);
    
    /**
    * Read a report: blocking
    * legacy method retained for backward compatability
    *
    * @param report pointer to the report to fill
    * @returns true if successful
    */
    bool read(HID_REPORT * report);
    
    /**
    * Read a report: non blocking
    * legacy method retained for backward compatability
    *
    * @param report pointer to the report to fill
    * @returns true if successful
    */
    bool readNB(HID_REPORT * report);
    
    /**
    * Provide the ability to Fill the inputReport. Report transfer is automatic if the data changes
    *
    * @param HID_REPORT
    * @returns true if succsful
    * The devloper should always use FillReport, The IDLE logic will send a report if the value has changed, or if the idle period expires.
    */
    bool FillInputReport(HID_REPORT *report);
    
    /**
    * Provide the ability to Fill the featureReport. 
    *
    * @param HID_REPORT
    * @returns true if succsful
    */
    bool FillFeatureReport(HID_REPORT *report);
    
    
    /**
    * Callback pointers to functions
    *
    * @param report pointer to the report to fill
    * These allow a user program to hook on get_report/set_report actions for each type.
    */
    void (*callbackSetInputReport)(HID_REPORT *report);
    /**
    * Callback pointers to functions
    *
    * @param report pointer to the report to fill
    * These allow a user program to hook on get_report/set_report actions for each type.
    */
    bool (*callbackGetInputReport)(HID_REPORT *report);
    /**
    * Callback pointers to functions
    *
    * @param report pointer to the report to fill
    * These allow a user program to hook on get_report/set_report actions for each type.
    */
    void (*callbackSetOutputReport)(HID_REPORT *report);
    /**
    * Callback pointers to functions
    *
    * @param report pointer to the report to fill
    * These allow a user program to hook on get_report/set_report actions for each type.
    */
    bool (*callbackGetOutputReport)(HID_REPORT *report);
    /**
    * Callback pointers to functions
    *
    * @param report pointer to the report to fill
    * These allow a user program to hook on get_report/set_report actions for each type.
    */
    void (*callbackSetFeatureReport)(HID_REPORT *report);
    /**
    * Callback pointers to functions
    *
    * @param report pointer to the report to fill
    * These allow a user program to hook on get_report/set_report actions for each type.
    */
    bool (*callbackGetFeatureReport)(HID_REPORT *report);
    
    /**
    * This function should be private, but is exposed to allow a developer to reenable IDLE values > zero after windows sets the value to zero. 
    *    This is due to microsoft windows HID driver not implementing IOCTL_HID_SET_POLL_FREQUENCY_MSEC/IOCTL_HID_GET_POLL_FREQUENCY_MSEC
    */
    virtual void SetReportIdle(uint16_t wValue);
    uint8_t protocolState;
    

protected:
    uint16_t reportLength;
    
    /*
    * Get the Report descriptor
    *
    * @returns pointer to the report descriptor
    */
    virtual uint8_t * reportDesc();

    /*
    * Get the length of the report descriptor
    *
    * @returns the length of the report descriptor
    */
    virtual uint16_t reportDescLength();

    /*
    * Get string product descriptor
    *
    * @returns pointer to the string product descriptor
    */
    virtual uint8_t * stringIproductDesc();
    
    /*
    * Get string interface descriptor
    *
    * @returns pointer to the string interface descriptor
    */
    virtual uint8_t * stringIinterfaceDesc();
    
    /*
    * Get configuration descriptor
    *
    * @returns pointer to the configuration descriptor
    */
    virtual uint8_t * configurationDesc();


    /*
    * HID Report received by SET_REPORT request. Warning: Called in ISR context
    * First byte of data will be the report ID
    *
    * @param report Data and length received
    */
    static void HID_callbackSetReport(HID_REPORT *report){};
    
    /*
    * HID Report received by GET_REPORT request. Warning: Called in ISR context
    * First byte of data will be the report ID
    *
    * @param report Data and length received
    */
    static bool HID_callbackGetReport(HID_REPORT *report){return true;};
   
    /*
    * Called by USBDevice on Endpoint0 request. Warning: Called in ISR context
    * This is used to handle extensions to standard requests
    * and class specific requests
    *
    * @returns true if class handles this request
    */
    virtual bool USBCallback_request();
    /*
    * Called by USBDevice on Endpoint0 request completion
    * if the 'notify' flag has been set to true. Warning: Called in ISR context
    *
    * In this case it is used to indicate that a HID report has
    * been received from the host on endpoint 0
    *
    * @param buf buffer received on endpoint 0
    * @param length length of this buffer
    */
    virtual void USBCallback_requestCompleted(uint8_t * buf, uint32_t length);

    /*
    * Called by USBDevice layer. to ensure that control transfers and interupt transfers work on the same reports.
    */
    virtual bool EP1_OUT_callback();
    
    /* Called by FillInputReport when data has changed, and transmits the data. */
    bool SendReport(void);
    
    /*
    * Called by USBDevice layer.to set configuration
    */
    virtual bool USBCallback_setConfiguration(uint8_t configuration);
    
    /* tells SET_IDLE/GET_IDLE where the IDE rate value is. */
    
    virtual uint8_t * getReportIdlePtr(uint8_t reportID){ return &reportIdleRateInt; };
    
    
private:
   // TODO: HID Reports should be dynamically malloced to save memory and prepare for multi-reports 
    HID_REPORT inputReport;
    HID_REPORT outputReport;
    HID_REPORT featureReport;
    uint8_t output_length;
    uint8_t input_length;
    uint8_t feature_length;
 
    //TODO: REPORT clsss should be created to handle MULTI-report
    //Implementation items for IDLE rate (TICKER)
    float reportIdleRate;
    uint8_t reportIdleRateInt;
    Ticker countDownToReportTrigger;
    void ResetIdleTicker(void);
    void IdlePeriodReSend(void);
   
    //Report REPORT data pointer retrieval.
    HID_REPORT * GetReportTargetPointer(uint16_t wValue);
    //Report Set/GET REPORT callback handlers
    bool HIDCallback_GetReportHandler(HID_REPORT *targetPtr);
    bool HIDCallback_SetReportHandler(HID_REPORT *targetPtr);
};

#endif
