// Copyright (c) 2013, jake (at) allaboutjake (dot) com
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * The name of the author and/or copyright holder nor the
//       names of its contributors may be used to endorse or promote products
//       derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER, AUTHOR, OR ANY CONTRIBUTORS
// BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
// GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT 
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
// OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

// DESCRIPTION OF FILE:
//
// This file contains a C++ object for communicating with a Makerbot over a USB
// host serial.
//

#include "makerbot.h"
#include "mbed.h"

// Constructor
Makerbot::Makerbot(USBHostSerial* hostSerial) {
    // Save the serial object
    serial = hostSerial;
    
    // Initialize caced values to NULL/zero, etc.
    machineName=NULL;
    buildName=NULL;
    machineVersion=0.0;
}

// Helper function to calculate the iButton CRC of a buffer
uint8_t Makerbot::CalculateCRC(uint8_t* data, int length) {
    uint8_t val = 0;
    for (int x=0; x<length; x++) { 
        val = crctab[val ^ data[x]];
    } 
    return val;
}

// Send a packet with the given payload.
int Makerbot::send_packet(uint8_t* payload, int payload_length) {
    // Verify payload length within bounds
    if (payload_length > MAXIMUM_PAYLOAD_LENGTH)
        return PACKET_TOO_BIG;
    
    // Calculate teh checksum    
    uint8_t checksum = CalculateCRC(payload, payload_length);
    
    // Send the data
    serial->putch(HEADER);
    serial->putch(payload_length);
    serial->writeBuffer(payload, payload_length);
    serial->putch(checksum);   
    
    // Return 0 for success;
    return 0;
}


// Receive a packet and fill the payload buffer with the packet's payload.
int Makerbot::recv_packet(uint8_t* payload, int buffer_length) {
    // Wait for a packet
    while (!serial->available()) { Thread::wait(100); }
    
    // Check for the header
    if (serial->getch() != HEADER) {
        DBG_INFO("recv_packet error: Header not found");
        return -1;
    }

    // Get the payload length
    int length = serial->getch();
    if (length > buffer_length) {
        DBG_INFO("recv_packet: Payload is larger then available buffer");
        return -1;
    }

    // Get the data with the given length
    for (int x=0; x<length; x++) {
        payload[x] = serial->getch();
    }
    
    // Get the checksum and verify the data we got matches the checksum
    int checksum = serial->getch();
    if (checksum < 0 || checksum != CalculateCRC(payload, length)) {
        DBG_INFO("recv_packet: Checksum error");
        return -1;
    } 
   
    // Return the length of the packet we received.
    return length;
}

// Display a message to to the Makerbot's LCD
int Makerbot::displayMessageToLCD(uint8_t options, uint8_t xPos, uint8_t yPos, uint8_t timeout, char*  message) {
    // Get the length of the message
    int msgLen = strlen(message);
    
    // The packet uses 5 bytes for the command and the other parameters
    // Make sure there is enough remaining space for the given message.
    if (msgLen > MAXIMUM_PAYLOAD_LENGTH - 5) {
        DBG_INFO("displayMessageToLCD: error: message too long.");
        return -1;
    }
    
    // Build the packet
    uint8_t packet[MAXIMUM_PAYLOAD_LENGTH+3];
    packet[0] = 149; // LCD
    packet[1] = options;
    packet[2] = xPos;
    packet[3] = yPos;
    packet[4] = timeout;
    for (int x=0; x<msgLen; x++) {
        packet[5+x] = message[x];
    }
    packet[5+msgLen] = 0x00; //terminate
    
    // Send the packet and receive the response
    mutex.lock();
    send_packet(packet, msgLen+6);    
    uint8_t response[MAXIMUM_PAYLOAD_LENGTH];
    recv_packet(response, MAXIMUM_PAYLOAD_LENGTH);
    mutex.unlock();
    
    // Return 0 if success (easier for error handling since SUCCESS is not zero)    
    if (response[0]==SUCCESS) return 0;
    
    // Otherwise return the actual response code.
    return response[0];
}

// Get the version of the makerbot
float Makerbot::getMakerbotVersion() {
    //If we have a cache of the version, just return it
    if (machineVersion) return machineVersion;
    
    //Build the packet
    uint8_t packet[3];
    packet[0]=0x00;
    packet[1]=0x00;
    packet[2]=0x64;
    
    //Send the packet and get the response packet
    mutex.lock();
    send_packet(packet, 3);       
    uint8_t response[MAXIMUM_PAYLOAD_LENGTH];   
    int length = recv_packet(response, MAXIMUM_PAYLOAD_LENGTH);
    mutex.unlock();
    
    //Make sure we got back the right payload length
    if (length != 3) {
        DBG_INFO("getMakerbotVersion: Expected 2 byte response. got %d", length);
        return 0; //Version=0 is an error in this case
    }
    
    //Convert the version to a float and return
    uint16_t versionData = *((uint16_t*)&response[1]);
    machineVersion = ((float)(versionData / 100)) + (((float)(versionData % 100)) / 10);
    return machineVersion;
}

char* Makerbot::getBuildName() {
    // One byte commend
    uint8_t cmd = 0x14; // 20 = Pause/Resume command

    // Send the Packet, get the response
    mutex.lock();    
    send_packet(&cmd, 1);
    uint8_t response[MAXIMUM_PAYLOAD_LENGTH];    
    int length = recv_packet(response, MAXIMUM_PAYLOAD_LENGTH);
    mutex.unlock();

    if (length > 1 && response[0]==SUCCESS) {
        //if we have an old build name, trash it
        if (buildName) {
            free(buildName);
            buildName = NULL;
        }
                
        buildName = (char*)malloc(length-1);
        strncpy(buildName, (char*)response+1,length-1);
        return buildName;
    }
    
    return NULL;
}

// Start a print job from the SD card with the given filename
int Makerbot::playbackCapture(char* filename) {   
    // Build the packet
    uint8_t packet[14];
    packet[0]=0x10; //Playback = 16
    
    // Copy over the filename into the packet one byte at a time
    // TODO: replace with memcpy?
    int length = 1;
    for (int x=0; x<12; x++) {
        packet[x+1] = filename[x];
        length++;
        if (filename[x]==0) break;
    }
    
    // Send the packet, get the response
    mutex.lock();
    send_packet(packet, length);           
    uint8_t response[MAXIMUM_PAYLOAD_LENGTH];
    int recvLength = recv_packet(response, MAXIMUM_PAYLOAD_LENGTH);
    mutex.unlock();
    
    // Check for response success
    if (recvLength > 0 && response[0] == SUCCESS) {
            return 0;
    }
    
    //If we got here, there was a problem somewhere
    return -1;
}

// Get machine name from bot's EEPROM
char* Makerbot::getMachineName() {
    // If we have a cached name, return it
    if (machineName) return machineName;
    
    // Build the request packet
    uint8_t payload[4];
    payload[0]= 0xC; //read eprom = 12
    *((uint16_t*)&payload[1])= MACHINE_NAME_OFFSET;    
    payload[3]= 16; //length of machine name;
   
    // Send the packet, wait for response.
    mutex.lock();
    send_packet(payload, 4);    
    uint8_t response[MAXIMUM_PAYLOAD_LENGTH];    
    int length = recv_packet(response, MAXIMUM_PAYLOAD_LENGTH);
    mutex.unlock();
    
    // Return
    if (length > 0 && response[0] == SUCCESS) {        
        //create a buffer to hold the machine name;
        machineName = (char*)malloc(length+1);
        
        memcpy(machineName, response+1, length-1);
        machineName[length-2] = '\0'; //terminate
        
        return machineName;
    }

    // Get machine name failed for some reason
    DBG_INFO("getMachineName: unsuccessful");
    return NULL;
}   

int Makerbot::getEEPROMByte(uint16_t offset) {     
    // Build the request packet
    uint8_t payload[4];
    payload[0]= 0xC; //read eprom = 12
    *((uint16_t*)&payload[1])= offset;    
    payload[3]= 1; //length of tool count (byte)
   
    // Send the packet, wait for response.
    mutex.lock();
    send_packet(payload, 4);    
    uint8_t response[MAXIMUM_PAYLOAD_LENGTH];    
    int length = recv_packet(response, MAXIMUM_PAYLOAD_LENGTH);
    mutex.unlock();
    
    // Return
    if (length == 2 && response[0] == SUCCESS) {               
        return response[1];
    }

    // Get machine name failed for some reason
    DBG_INFO("getEEPROMByte: unsuccessful");
    return 0;
}

// Get tool count from bot's EEPROM
int Makerbot::getToolCount() { 
    return getEEPROMByte(TOOL_COUNT_OFFSET);
}

int Makerbot::hasPlatform() {
    return getEEPROMByte(HBP_PRESENT_OFFSET);
}

// Flush the buffer.  Used to synchronize the stream
void Makerbot::flushInputChannel() {
    mutex.lock();
    while (1) {
        if (serial->available()) 
            serial->getch();
        else
            break;
    }
    mutex.unlock();
}

// Send pause/resume command
int Makerbot::pauseResume() {
    // One byte commend
    uint8_t cmd = 0x08; // 8 = Pause/Resume command

    // Send the Packet, get the response
    mutex.lock();    
    send_packet(&cmd, 1);
    uint8_t response[MAXIMUM_PAYLOAD_LENGTH];    
    int length = recv_packet(response, MAXIMUM_PAYLOAD_LENGTH);
    mutex.unlock();
    
    // Return success status
    if (length == 1 && response[1] == SUCCESS) return 0;
    return -1;
}

// Send command to cancel build command
int Makerbot::abortImmediately() {
    // One byte commend
    uint8_t cmd = 0x07; // 7 = Abort command

    // Send the Packet, get the response
    mutex.lock();    
    send_packet(&cmd, 1);
    uint8_t response[MAXIMUM_PAYLOAD_LENGTH];    
    int length = recv_packet(response, MAXIMUM_PAYLOAD_LENGTH);
    mutex.unlock();
    
    // Return success status
    if (length == 1 && response[1] == SUCCESS) return 0;
    return -1;
}

// Retrieve the current temperature for the given tool
int Makerbot::getToolTemperature(uint8_t toolIndex) {
    // One byte commend
    uint8_t payload[3];
    payload[0] = 0x0A; // 7 = Abort command
    payload[1] = toolIndex;
    payload[2] = 0x2; //get toolhead temperature
    
    // Send the Packet, get the response
    mutex.lock();    
    send_packet(payload, 3);
    uint8_t response[MAXIMUM_PAYLOAD_LENGTH];    
    int length = recv_packet(response, MAXIMUM_PAYLOAD_LENGTH);
    mutex.unlock();
    
   
    // Return success status
    if (length == 3 && response[0] == SUCCESS) {
        return *((uint16_t*)&response[1]);
    }
    return -1;
}

// Return the setpoint for the given tool
int Makerbot::getToolSetPoint(uint8_t toolIndex) {
    // One byte commend
    uint8_t payload[3];
    payload[0] = 0x0A; // 7 = Abort command
    payload[1] = toolIndex;
    payload[2] = 0x20; //get toolhead set point
    
    // Send the Packet, get the response
    mutex.lock();    
    send_packet(payload, 3);
    uint8_t response[MAXIMUM_PAYLOAD_LENGTH];    
    int length = recv_packet(response, MAXIMUM_PAYLOAD_LENGTH);
    mutex.unlock();
    
   
    // Return success status
    if (length == 3 && response[0] == SUCCESS) {
        return *((uint16_t*)&response[1]);
    }
    return -1;
}


// Retrieve the Platform temperature for the given tool
int Makerbot::getPlatformTemperature(uint8_t toolIndex) {
    // One byte commend
    uint8_t payload[3];
    payload[0] = 0x0A; // 7 = Abort command
    payload[1] = toolIndex;
    payload[2] = 0x1E; //get toolhead temperature
    
    // Send the Packet, get the response
    mutex.lock();    
    send_packet(payload, 3);
    uint8_t response[MAXIMUM_PAYLOAD_LENGTH];    
    int length = recv_packet(response, MAXIMUM_PAYLOAD_LENGTH);
    mutex.unlock();
    
   
    // Return success status
    if (length == 3 && response[0] == SUCCESS) {
        return *((uint16_t*)&response[1]);
    }
    return -1;
}

// Return the platform setpoint for the given tool
int Makerbot::getPlatformSetPoint(uint8_t toolIndex) {
    // One byte commend
    uint8_t payload[3];
    payload[0] = 0x0A; // 7 = Abort command
    payload[1] = toolIndex;
    payload[2] = 0x21; //get toolhead set point
    
    // Send the Packet, get the response
    mutex.lock();    
    send_packet(payload, 3);
    uint8_t response[MAXIMUM_PAYLOAD_LENGTH];    
    int length = recv_packet(response, MAXIMUM_PAYLOAD_LENGTH);
    mutex.unlock();
    
   
    // Return success status
    if (length == 3 && response[0] == SUCCESS) {
        return *((uint16_t*)&response[1]);
    }
    return -1;
}

// Get the build state.  Note that we added a "BUILD_STATE_ERROR" value to detect errors in the return value.
Makerbot::MakerbotBuildState Makerbot::getBuildState() {
    MakerbotBuildState state;
    
    // One byte command
    uint8_t cmd = 0x18; //24=get build statistics    
    
    // Send the packet, get the response
    mutex.lock();    
    send_packet(&cmd, 1);
    uint8_t response[MAXIMUM_PAYLOAD_LENGTH];
    int length = recv_packet(response, MAXIMUM_PAYLOAD_LENGTH);
    mutex.unlock();
    
    
    // Check for success
    if (response[0] == SUCCESS) {
        // Copy the data to the return struct & return
        memcpy(&state, response+1, sizeof(MakerbotBuildState));
        return state;
    }
    
    // Otherwise return error state
    state.build_state = BUILD_STATE_ERROR; //error
    return state; 
}

char* Makerbot::humanReadableBuildState(Makerbot::MakerbotBuildState state) {   
    switch (state.build_state) {                        
            case Makerbot::NO_BUILD:
                return "No Build";

            case Makerbot::BUILD_RUNNING:
                return "Build Running";                
            
            case Makerbot::BUILD_FINISHED_NORMALLY:
                return "Build Finished";                
            
            case Makerbot::BUILD_PAUSED:
                return "Build Paused";                
            
            case Makerbot::BUILD_CANCELLED:
                return "Build Cancelled";                
            
            case Makerbot::BUILD_SLEEPING:
                return "Build Sleeping";                
                       
            case Makerbot::BUILD_STATE_ERROR:
            default:
                return "Unknown build status or status error";                
    }
}

// Clean up any objects created, just for good measure.
Makerbot::~Makerbot() {
    if (machineName) {
        free(machineName);
    }
    if (buildName) {
        free(buildName);
    }
}