// Copyright (c) 2013, jake (at) allaboutjake (dot) com
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * The name of the author and/or copyright holder nor the
//       names of its contributors may be used to endorse or promote products
//       derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER, AUTHOR, OR ANY CONTRIBUTORS
// BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
// GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT 
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
// OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

// DESCRIPTION OF FILE:
//
// This is the associated header file and contains the declaration of a C++ 
// object for communicating with a Makerbot over a USB host serial.
//

#include "mbed.h"
#include "USBHostSerial.h"
#include "rtos.h"

#ifndef __MAKERBOT_H__
#define __MAKERBOT_H__

typedef unsigned char uint8_t;

#define HEADER 0xD5
#define MAXIMUM_PAYLOAD_LENGTH 32
#define MAXIMUM_PACKET_LENGTH (MAXIMUM_PAYLOAD_LENGTH+3)

// Response Codes
#define HEADER_NOT_FOUND 0xFF
#define GENERIC_PACKET_ERROR 0x80
#define SUCCESS 0x81 
#define ACTION_BUFFER_OVERFLOW 0x82 
#define CRC_MISMATCH 0x83 
#define PACKET_TOO_BIG 0x84
#define COMMAND_NOT_SUPPORTED 0x85 
#define DOWNSTREAM_TIMEOUT 0x87 
#define TOOL_LOCK_TIMEOUT 0x88 
#define CANCEL_BUILD 0x89 
#define ACTIVE_LOCAL_BUILD 0x8A 
#define OVERHEAT_STATE 0x8B 

// EEPROM Offsets
#define MACHINE_NAME_OFFSET 0x0022
#define TOOL_COUNT_OFFSET 0x0042
#define HBP_PRESENT_OFFSET 0x004C

const uint8_t crctab[] = {
   0, 94, 188, 226, 97, 63, 221, 131, 194, 156, 126, 32, 163, 253, 31, 65,
   157, 195, 33, 127, 252, 162, 64, 30, 95, 1, 227, 189, 62, 96, 130, 220,
   35, 125, 159, 193, 66, 28, 254, 160, 225, 191, 93, 3, 128, 222, 60, 98,
   190, 224, 2, 92, 223, 129, 99, 61, 124, 34, 192, 158, 29, 67, 161, 255,
   70, 24, 250, 164, 39, 121, 155, 197, 132, 218, 56, 102, 229, 187, 89, 7,
   219, 133, 103, 57, 186, 228, 6, 88, 25, 71, 165, 251, 120, 38, 196, 154,
   101, 59, 217, 135, 4, 90, 184, 230, 167, 249, 27, 69, 198, 152, 122, 36,
   248, 166, 68, 26, 153, 199, 37, 123, 58, 100, 134, 216, 91, 5, 231, 185,
   140, 210, 48, 110, 237, 179, 81, 15, 78, 16, 242, 172, 47, 113, 147, 205,
   17, 79, 173, 243, 112, 46, 204, 146, 211, 141, 111, 49, 178, 236, 14, 80,
   175, 241, 19, 77, 206, 144, 114, 44, 109, 51, 209, 143, 12, 82, 176, 238,
   50, 108, 142, 208, 83, 13, 239, 177, 240, 174, 76, 18, 145, 207, 45, 115,
   202, 148, 118, 40, 171, 245, 23, 73, 8, 86, 180, 234, 105, 55, 213, 139,
   87, 9, 235, 181, 54, 104, 138, 212, 149, 203, 41, 119, 244, 170, 72, 22,
   233, 183, 85, 11, 136, 214, 52, 106, 43, 117, 151, 201, 74, 20, 246, 168,
   116, 42, 200, 150, 21, 75, 169, 247, 182, 232, 10, 84, 215, 137, 107, 53
};


class Makerbot { 

public:

    enum LCDOptions {
        CLEAR_EXISTING_MESSAGE = 0x01,
        LAST_MESSAGE_IN_GROUP = 0x02,
        WAIT_FOR_BUTTON_PRESS = 0x04,     
    };

    enum BuildState {
        NO_BUILD = 0,
        BUILD_RUNNING = 1,
        BUILD_FINISHED_NORMALLY = 2,
        BUILD_PAUSED = 3,
        BUILD_CANCELLED = 4,
        BUILD_SLEEPING = 5,
        BUILD_STATE_ERROR = 0xFF,
    };
    
    typedef struct {
        uint8_t build_state;
        uint8_t hours;
        uint8_t minutes;
        uint32_t line;
        uint32_t reserved;
    } __attribute__((packed, aligned(1))) MakerbotBuildState;
    
    // Constructor / Destructor
    Makerbot(USBHostSerial* serial);
    ~Makerbot();
    
    int displayMessageToLCD(uint8_t options, uint8_t xPos, uint8_t yPos, uint8_t timeout, char*  message);
    float getMakerbotVersion();
    char* getMachineName();
    int playbackCapture(char* filename);  
    int pauseResume();
    int abortImmediately();
    int getToolCount();
    int getToolTemperature(uint8_t toolIndex);
    int getToolSetPoint(uint8_t toolIndex);
    int getPlatformTemperature(uint8_t toolIndex);
    int getPlatformSetPoint(uint8_t toolIndex);
    char* getBuildName();
    int hasPlatform();
    MakerbotBuildState getBuildState();
   
    void flushInputChannel();
    
    static char* humanReadableBuildState(Makerbot::MakerbotBuildState state);
    
private:
    uint8_t CalculateCRC(uint8_t* data, int length);
    int getEEPROMByte(uint16_t offset);
    
    int send_packet(uint8_t* payload, int length);
    int recv_packet(uint8_t* payload, int buffer_length);
  
    USBHostSerial* serial;
    Mutex mutex;
    
    char* machineName;
    float machineVersion;
    char* buildName;
    
};


#define DBG_INFO(x, ...) std::printf("[DBG_INFO: %s:%d]"x"\r\n", __FILE__, __LINE__, ##__VA_ARGS__);

#endif