// Copyright (c) 2013, jake (at) allaboutjake (dot) com
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * The name of the author and/or copyright holder nor the
//       names of its contributors may be used to endorse or promote products
//       derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER, AUTHOR, OR ANY CONTRIBUTORS
// BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
// GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT 
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
// OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

// DESCRIPTION OF FILE:
//
// This file contains a simple telnet server on port 7654.   The main function is
// "telnet_server_task" which is intended to be run in a thread after the 
// network (EthernetInterface) bas been configured.
#include "mbed.h"
#include "main.h"
#include "telnetd.h"
#include "EthernetInterface.h"
#include "SimpleSocket.h"
#include "readline.h"
#include "settings.h"

int readline_password(ClientSocket* socket, char* buffer, int max_length);

// Network server task: setup and listen for network connections
void telnet_server_task(void const*) {
    // This thread assumes that the EthernetInterface is connected with an IP address.
    static int TELNET_PORT = 7654;

    //Use SimpleSocket library to setup a server socket
    ServerSocket serverSocket(TELNET_PORT);
    
    while (1) {
        ClientSocket socket = serverSocket.accept();    
    
        //Create buffer to hold the command prompt input and filename
        char promptBuffer[MAX_PROMPT_BUFFER_SIZE];
        
        char* token = getAuthenticationToken();
        if (token != NULL) { 
            int authLen = readline_password(&socket, promptBuffer, MAX_PROMPT_BUFFER_SIZE);            
            if (memcmp(promptBuffer, token, authLen) != 0) {
                socket.printf("Authentication failed\r\n");
                socket.close();
                continue;
            }
        }
        
        //Main loop
        while (1) {
            // Print the prompt        
            socket.printf("> ");
        
            // Get the command
            int cmdLength = readline(&socket, promptBuffer, MAX_PROMPT_BUFFER_SIZE);
                
            if (cmdLength > 0) {
                switch(promptBuffer[0]) {
                    case 'A':
                    case 'a': 
                    {
                        int tokenLength = cmdLength - 2;
                        if (tokenLength > 1) {
                            setAuthenticationToken(promptBuffer+2, tokenLength);
                            char* token = getAuthenticationToken();
                            socket.printf("Authentication set to: %s\r\n", token);                            
                        } else {
                            socket.printf("Command usage: A <username>:<password> - set authentication username/password\r\n");
                        }
                        break;
                    }
                        
                    case 'B':
                    case 'b':
                        //BUILD command - make sure there's a space between cmd and filename
                        if (cmdLength > 1 && promptBuffer[1]==' ') {
                            int fileNameLength = cmdLength - 2;
                            //make sure the filename is at least 5 characters (minimally "?.X3G")
                            if (fileNameLength >= 5) {
                                // Check for X3G extension.
                                if (memcmp(promptBuffer+cmdLength-4, ".X3G", 4)==0 ||
                                    memcmp(promptBuffer+cmdLength-4, ".x3g", 4)==0) {
                                    // Copy filename into filename buffer
                                    char filename[13];
                                    strncpy(filename, promptBuffer+2, cmdLength-2);
                                    filename[cmdLength-2]='\0';//terminate
                                
                                    //Print the message to the display and the serial console
                                    DISPLAY("Printing filename: '%s'\r\n", filename);
                                    socket.printf("Printing filename: '%s'\r\n", filename);
                                
                                    //Send command to bot to print
                                    if (bot) {
                                        bot->playbackCapture(filename);
                                    } else {                                   
                                        socket.printf("Error: connection to Makerbot not established.\r\n");                                    
                                    }
                                } else {
                                    socket.printf("Error: filename doesn't end with X3G\r\n");
                                }
                            } else {
                                socket.printf("Error: filename must be 5-12 characters long\r\n");
                            }
                        } else {
                            socket.printf("Command usage: B <filename.X3G> - begin build (8.3 filename)\r\n");
                            
                        }
                        break;
                    case 'C':
                    case 'c':
                        if (bot) {
                            bot->abortImmediately();
                            socket.printf("Cancelling build.\r\n");
                        } else {
                            socket.printf("Error: connection to Makerbot not established.\r\n");                                    
                        }
                        break;
                    
                    case 'P':
                    case 'p':
                        if (bot) {
                            Makerbot::MakerbotBuildState state = bot->getBuildState();
                            if (state.build_state == Makerbot::BUILD_STATE_ERROR) {
                                socket.printf("Error: unable to determine build state.  Sending pause/resume command in the blind.\r\n");
                                bot->pauseResume();
                            } else if (state.build_state == Makerbot::BUILD_RUNNING) {
                                bot->pauseResume();
                                socket.printf("Pausing build.\r\n");
                            } else {
                                socket.printf("Error: Not currently building, cannot pause\r\n");
                            }
                        } else {
                            socket.printf("Error: connection to Makerbot not established.\r\n");                                    
                        }
                        break;

                    case 'R':
                    case 'r':
                        if (bot) {
                            Makerbot::MakerbotBuildState state = bot->getBuildState();
                            if (state.build_state == Makerbot::BUILD_STATE_ERROR) {
                                socket.printf("Error: unable to determine build state.  Sending pause/resume command in the blind.\r\n");
                                bot->pauseResume();
                            } else if (state.build_state == Makerbot::BUILD_PAUSED) {
                                bot->pauseResume();
                                socket.printf("Pausing build.\r\n");
                            } else {
                                socket.printf("Error: Not currently paused.  Cannot resume.\r\n");
                            }
                        } else {
                            socket.printf("Error: connection to Makerbot not established.\r\n");                                    
                        }
                        break;

                    case 'S':
                    case 's':
                        if (bot) {
                            Makerbot::MakerbotBuildState state = bot->getBuildState();
                            socket.printf("Build State: %s\r\n", Makerbot::humanReadableBuildState(state));
                            socket.printf("\tElapsed build time: %d:%02d\r\n", state.hours, state.minutes);
                            socket.printf("\tLine #: %ld\r\n", state.line);
                            
                            for (int x=0; x<bot->getToolCount(); x++) {                            
                                socket.printf("\tTool %d: %dºc/%dºc\r\n", x, bot->getToolTemperature(x), bot->getToolSetPoint(x));
                            }
                            socket.printf("\tPlatform %dºc/%dºc\r\n", bot->getPlatformTemperature(0), bot->getPlatformSetPoint(0));

                        }
                        break;
                    
                    case 'Q':
                    case 'q':
                        socket.printf("Logging out\r\n");
                        socket.close();                        
                        break;
                        
                    case 'v':
                    case 'V': 
                        if (bot) {
                            char* bot_name = bot->getMachineName();   
                            socket.printf("Machine name: %s\r\n", bot_name);
    
                            float version = bot->getMakerbotVersion();
                            socket.printf("Makerbot version: %0.1f\r\n", version);
                            socket.printf("Local firmware version: %s\r\n", FIRMWARE_VERSION);                            
                        } else {
                            socket.printf("Error: connection to Makerbot not established.\r\n");
                        }
                        break;
                    default:      
                        socket.printf("unknown command\r\n");                          
                }
            }                          
            Thread::wait(100);        

            // Go back around and wait for a new connection if we've disconnected
            if (!socket.connected())
                break;
        }
    }
    // never gets here
    //EthernetInterface::disconnect();  
}

//Readline routine customized to return a "username:password" style string
int readline_password(ClientSocket* socket, char* buffer, int max_length) {
   int curPos = 0;
   int usernamePasswordStep = 0;
   socket->printf("user: ");
   
   while (1) {
        int c;
        
        //Wait for a character
        while ((c=socket->read()) < 0) {
            if (!socket->connected()) return -1;
            Thread::yield();
        }
            
        switch (c) {
            case '\r':
                continue;
                                    
            case '\b':
                if (curPos > 0) curPos--;
                break; 
                
            default:
                buffer[curPos]= (char)(c & 0xFF);
                curPos++;
                if (curPos < max_length)
                    break;
                    
                //FALLTHROUGH
            case '\n':
                if (usernamePasswordStep==0) {
                    socket->printf("password: ");
                    buffer[curPos] = ':';
                    curPos++;
                    usernamePasswordStep++;
                    break;
                } else {
                    socket->printf("\r\n");
                    buffer[curPos]=0x00;  //string terminator
                    return curPos;
                }
        }
    } 
    
}
