// Copyright (c) 2013, jake (at) allaboutjake (dot) com
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * The name of the author and/or copyright holder nor the
//       names of its contributors may be used to endorse or promote products
//       derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER, AUTHOR, OR ANY CONTRIBUTORS
// BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
// GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT 
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
// OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

// DESCRIPTION OF FILE:
//
// This file contains the main entry point for the firmware.  It sets up 
// the necessary hardware peripherals and starts the application threads.
//


#include "main.h"
#include "stdio.h"
#include "mbed.h"
#include "Mini_DK.h"
#include "EthernetInterface.h"
#include "NTPClient.h"
#include "RTC.h"
#include "USBHostSerial.h"
#include "makerbot.h"
#include "SimpleSocket.h"
#include "readline.h"
#include "telnetd.h"
#include "httpd.h"
#include "NTPClient.h"
#include "NTPClient.h"

#ifdef TARGET_MINIDK
    //Define the pins and objects related to the Mini_DK2 development board
    DigitalOut led1(DK_LED1);
    DigitalOut led2(DK_LED2);
    DigitalOut usbpwr(USB_PPWR);

    InterruptIn key1(DK_KEY1);
    InterruptIn key2(DK_KEY2);
#ifdef MINIDK_HAS_LCD
    // TFT -> mosi, miso, sclk, cs
    SPI_TFT TFT(LCD_SDI, LCD_SDO, LCD_SCK, LCD_CS,"TFT");

    // ADS7843 -> mosi, miso, sclk, cs, irq, SPI_TFT
    //TouchScreenADS7843 TP(TP_SDI ,TP_SDO ,TP_SCK ,TP_CS ,TP_IRQ, &TFT);
#endif

#endif

#ifdef TARGET_MBED
    DigitalOut led1(LED1);
    DigitalOut led2(LED2);
#endif

//Declare the serial interface
Serial pc(USBTX, USBRX);

//Declare the USBHostSerial and Makerbot object pointers
USBHostSerial* serial;
Makerbot* bot;

//function to initialize the LCD (for Mini-DK2 only)
void initialize_TFT() {
#if defined(TARGET_MINIDK) && defined(MINIDK_HAS_LCD)
    //TFT.claim(stdout);        // send stdout to the TFT display
    TFT.background(Black);    // set background to black
    TFT.foreground(White);    // set chars to white
    
    TFT.cls();
    TFT.set_orientation(1);
    TFT.set_font((unsigned char*) Arial12x12);
    TFT.locate(0,0);
#endif
}

// Configure and wait for a the USB cable to be connected to the Makerbot
void serial_task(void const*) {
    //Create the USBHostSerial object    
    serial = new USBHostSerial;
    
    DISPLAY("Initializing USBHostSerial.\r\n");
    DISPLAY("Trying to connect to serial device.\r\n");
        
    // try to connect a serial device.  Wait and retry if no connection
    while(!serial->connect())
        Thread::wait(500);

    //Set N,8,1
    DISPLAY("Setting serial parameters.\r\n");
    serial->format(8, USBHostSerial::None, 1);
    
    //Set baud to 115200 (as is used by the Makerbot)
    DISPLAY("Setting baud to 115200\r\n");
    serial->baud(115200);
    
    //All done!
    DISPLAY("Serial initialized.\r\n");
    
    //Create the bot object for interacting with the machine over the s3g protocol
    bot = new Makerbot(serial);
    
    //Quick synchronizaton of the packet stream
    bot->getMakerbotVersion();
    bot->flushInputChannel();
    
    //Get the version of the makerbot
    float version = bot->getMakerbotVersion();
    DISPLAY("Makerbot version: %0.1f\r\n", version);
    
    //Get the bot name
    char* bot_name = bot->getMachineName();   
    DISPLAY("Machine name: %s\r\n", bot_name);
    
}




// A pointer for the task that acts on the button interrupts
Thread* buttonTask;

// Interrupt handler for the first button
void isr_key1() {
    //Set a signal 0b00000001 = button 1.
    buttonTask->signal_set(0x1);
}
 
// Interrupt handler for the second button
void isr_key2() {
    //Set a signal 0b00000010 = button 2.
    buttonTask->signal_set(0x2);
}

//The task that handles the button presses
void button_task(void const*) {
    while (1) {
        //Wait for any signal to this thread
        osEvent evt = Thread::signal_wait(0);
        
        // If button=1 then send the pause/resume command;        
        if (evt.value.signals & 0x01) {
            if (bot) {
                Makerbot::MakerbotBuildState state = bot->getBuildState();
                if (state.build_state == Makerbot::BUILD_STATE_ERROR) {
                    //DISPLAY("Error: unable to determine build state.  Sending pause/resume command in the blind.\r\n");
                    bot->pauseResume();
                } else if (state.build_state == Makerbot::BUILD_RUNNING) {
                    bot->pauseResume();
                    //DISPLAY("Pausing build.\r\n");
                } else {
                    //DISPLAY("Error: Not currently building, cannot pause\r\n");
                }
            } else {
               //DISPLAY("Error: connection to Makerbot not established.\r\n");                                    
            }
        }
        
        // If button=2 print a message to the LCD (debug purposes)
        //TODO: replace this with something useful
        if (evt.value.signals & 0x02) {
            if (bot) {
                Makerbot::MakerbotBuildState state = bot->getBuildState();
                if (state.build_state == Makerbot::BUILD_STATE_ERROR) {
                    //DISPLAY("Error: unable to determine build state.  Sending pause/resume command in the blind.\r\n");
                    bot->pauseResume();
                } else if (state.build_state == Makerbot::BUILD_PAUSED) {
                    bot->pauseResume();
                    //DISPLAY("Pausing build.\r\n");
                } else {
                    //DISPLAY("Error: Not currently paused.  Cannot resume.\r\n");
                }
            } else {
                //DISPLAY("Error: connection to Makerbot not established.\r\n");                                    
            }
            break;
        }        
        Thread::wait(250);
    }
}
 

int main() {
    //Set the baud rate off the serial debug console.
    pc.baud(115200);
    pc.printf("Firmware Version: %s\r\n", FIRMWARE_VERSION);
     
#ifdef TARGET_MINIDK  

#ifdef MINIDK_HAS_LCD
    //Initialize the LCD
    initialize_TFT();
    DISPLAY("Firmware Version: %s\r\n", FIRMWARE_VERSION);
#endif
    //Attach the ISR's to the buttons
    key1.fall(isr_key1);
    key2.fall(isr_key2);

    //Start the task that watches for button press signals
    buttonTask = new Thread(button_task, NULL, osPriorityNormal, 128 * 4);
    
    usbpwr=1; //turn on power to the USBHost port.
#endif

    // Print the MAC address for reference
    char mac[6];
    mbed_mac_address(mac);
    DISPLAY("MAC ADDRESS: %02x:%02x:%02x:%02x:%02x:%02x\r\n", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]); 

    
    DISPLAY("Negotiating network.\r\n");
    EthernetInterface::init();
    EthernetInterface::connect();
    DISPLAY("IP Address: %s\r\n", EthernetInterface::getIPAddress());
    
    DISPLAY("Attempting to set clock...\r\n");
    NTPClient* ntp = new NTPClient;
    if (ntp->setTime("0.pool.ntp.org") == 0) {      
      time_t ctTime;
      ctTime = time(NULL);
      DISPLAY("Time is set to (UTC): %s\r\n", ctime(&ctTime));
    } else {
      printf("Unable to set time\r\n");
    }
    delete ntp;
    
    // Turn on LED1 (LED1 will blink once we get to main loop)
    led1=1;
    bot = NULL; //Initialize bot object to NULL so we know when there's a bot object by testing for null
    
    //Setup the serial and server tasks
    Thread serialTask(serial_task, NULL, osPriorityNormal, 256 * 4); 
    Thread serverTask(telnet_server_task, NULL, osPriorityNormal, 512 * 4);     
    Thread serverTask2(http_server_task, NULL, osPriorityNormal, 512 * 4);     
    
    //Main loop-- blink the LED and yield to other tasks during WAIT.
    while(1) {    
        led1=!led1;
        Thread::wait(500);
    }
}