// Copyright (c) 2013, jake (at) allaboutjake (dot) com
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * The name of the author and/or copyright holder nor the
//       names of its contributors may be used to endorse or promote products
//       derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER, AUTHOR, OR ANY CONTRIBUTORS
// BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
// GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT 
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT 
// OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

// DESCRIPTION OF FILE:
//
// This file contains the source to the http server.
//
// The main function is "http_server_task" which is intended to be run in its
// own thread.  The other functions are helper functions.  This function assumes
// that the EthernetInterface is already configured and ready for use.
//

#include "httpd.h"
#include "readline.h"
#include "main.h"
#include "settings.h"

// Some Private methods
// HTML rendering
void sendHTTPHeader(ClientSocket* sock, int response_code, char* response_descr, char* contentType);
void sendStatusContent(ClientSocket* sock);
void sendPageHeader(ClientSocket* sock);
void sendPageFooter(ClientSocket* sock);
void sendCommandTable(ClientSocket* sock);
void sendHTTPRedirect(ClientSocket* sock, char* redirectTo);
void redirectToMessage(ClientSocket* sock, char* message, int isErr);
void renderMessageBox(ClientSocket* sock, char* message, int isBadMessage);
int handleAuthorizationHeaders(ClientSocket* sock, char* token);
void discardHeaders(ClientSocket* socket);

//Helper functions
int findOccuranceOfChar(char* buffer, char c, int occur, int max_length);
char *url_encode(char *str);
char *url_decode(char *str);

// Network server task: setup and listen for network connections
void http_server_task(void const*) {
    //This thread assumes that the EthernetInterface is connected with an IP address.       
    static int HTTPD_PORT = 80;
    
    //Use SimpleSocket library to setup a server socket
    DISPLAY("Setting up httpd on %d\r\n", HTTPD_PORT);    
    ServerSocket serverSocket(HTTPD_PORT);
    
    while (1) {
        //Debug: DISPLAY("Waiting for connection\r\n");
        //Single-threaded: get the next incoming connection
        ClientSocket socket = serverSocket.accept();            

retryAuth:         
        //Get the request line (the first line) of the HTTP request
        char linebuffer[HTTP_LINE_BUFFER_SIZE];
        memset(linebuffer, 0, HTTP_LINE_BUFFER_SIZE);
        int length = readline(&socket, linebuffer, HTTP_LINE_BUFFER_SIZE);
         
        //debug:  printf("%s: '%s'\r\n", socket.get_address(), linebuffer);
               
        // Make sure its a GET (we only support GET currently)
        if (memcmp(linebuffer, "GET ", 4) != 0) {
            printf("Request Line: '%s'\r\n", linebuffer);            
            DISPLAY("SENDING 400 Bad Request\r\n");
            sendHTTPHeader(&socket, 400, "Bad Request", "text/html"); 
            sendPageHeader(&socket);
            socket.printf("400 Bad Request");
            sendPageFooter(&socket);  
            socket.close();            
            continue;
        }       
        
        // If we have a password stored in flash, then check to see if we're authorized.  
        char* token = getAuthenticationToken();
        if (token != NULL) { 
            //Process the headers, look for an authoirzation header and discard the rest.
            char* base64token=base64(token);
            int authState = handleAuthorizationHeaders(&socket, base64token);
            free(base64token);
            discardHeaders(&socket);
            
            // If authentication was not good, then send out the 401
            if (!authState) {
                //Auth failed or not present.
                socket.printf("HTTP/1.1 %d %s\r\n", 401, "Not Authorized");
                socket.printf("WWW-Authenticate: Basic realm=\"makerbot_server\"\r\n");
                socket.printf("\r\n");                                  
                Thread::yield();
                goto retryAuth;  //TODO: goto's bad, but I'm lazy. Can probably be refactored somehow                
            } else {
                //Authentication OK
            }
        } else {
            //A password wasn't set in flash, we don't need the rest of the headers, discard
            discardHeaders(&socket);
        }        
        
        //Find the path section of the request line.
        int endOfPath = findOccuranceOfChar(linebuffer, ' ', 2, HTTP_LINE_BUFFER_SIZE);
        int lengthOfPath = endOfPath - 4;
        char* path = (char*)malloc(lengthOfPath+1);
        strncpy(path, linebuffer+4, lengthOfPath);
        path[lengthOfPath]=0;
        
        //debugging: DISPLAY("PATH (%d): '%s'\r\n", lengthOfPath, path);
        
        // Process the given path and return the page       
        if (lengthOfPath == 1 && path[0] == '/') {
            //request for the root (default) page
            sendHTTPHeader(&socket, 200, "OK", "text/html");
            sendPageHeader(&socket);            
            sendStatusContent(&socket);
            sendCommandTable(&socket);
            sendPageFooter(&socket);
        } else if (lengthOfPath > 6 && memcmp(path, "/?msg=",6) == 0) {
            char* message = url_decode(path+6);
            sendHTTPHeader(&socket, 200, "OK", "text/html");
            sendPageHeader(&socket);
            renderMessageBox(&socket, message, 0);
            sendStatusContent(&socket);
            sendCommandTable(&socket);
            sendPageFooter(&socket);            
            free(message);
        } else if (lengthOfPath > 6 && memcmp(path, "/?err=",6) == 0) {
            char* message = url_decode(path+6);
            sendHTTPHeader(&socket, 200, "OK", "text/html");
            sendPageHeader(&socket);
            renderMessageBox(&socket, message, 1);
            sendStatusContent(&socket);
            sendCommandTable(&socket);
            sendPageFooter(&socket);            
            free(message);
        } else if (strncmp(path, "/command?", 9) == 0) {
            //This is a command, figure out which one                          
            if (lengthOfPath >= 9+5 && memcmp(path+9, "pause", 5) == 0) {
                //pause command
                //GET /command?pause HTTP/1.1 
                if (bot) {
                    Makerbot::MakerbotBuildState state = bot->getBuildState();
                    if (state.build_state == Makerbot::BUILD_STATE_ERROR) {
                        redirectToMessage(&socket, "Error: unable to determine build state.  Sending pause/resume command in the blind.", 1);
                        bot->pauseResume();
                    } else if (state.build_state == Makerbot::BUILD_RUNNING) {
                        bot->pauseResume();
                        redirectToMessage(&socket, "Pausing build.", 0);
                    } else {
                        redirectToMessage(&socket, "Error: Not currently running.  Cannot pause.", 1);
                    }
                } else {
                    redirectToMessage(&socket, "Error: connection to Makerbot not established.", 1);
                }                                
            } else if (lengthOfPath >= 9+6 && memcmp(path+9, "cancel", 6) == 0) {
                if (bot) {
                    bot->abortImmediately();
                    redirectToMessage(&socket, "Cancelling build.", 0);
                } else {
                    redirectToMessage(&socket, "Error: connection to Makerbot not established.", 1);                                    
                }                                
            } else if (lengthOfPath >= 9+6 && memcmp(path+9, "resume", 6) == 0) {                
                if (bot) {
                    Makerbot::MakerbotBuildState state = bot->getBuildState();
                    if (state.build_state == Makerbot::BUILD_STATE_ERROR) {
                        redirectToMessage(&socket, "Error: unable to determine build state.  Sending pause/resume command in the blind.", 1);
                        bot->pauseResume();
                    } else if (state.build_state == Makerbot::BUILD_PAUSED) {
                        bot->pauseResume();
                        redirectToMessage(&socket, "Resuming build.", 0);
                    } else {
                        redirectToMessage(&socket, "Error: Not currently paused.  Cannot resume.", 1);
                    }
                } else {
                    redirectToMessage(&socket, "Error: connection to Makerbot not established.", 1);
                }                
            } else if (lengthOfPath >= 9+6 && memcmp(path + 9, "build=", 6) == 0) {
                int startOfFilename = findOccuranceOfChar(path, '=', 1, lengthOfPath)+1;
                int lengthOfFileName = lengthOfPath - startOfFilename;
                char* filename = path+startOfFilename;
                if (lengthOfFileName < 5) {
                    //Minimum filename is of the form "x.x3g" = 5 characters
                    redirectToMessage(&socket, "Filename too short", 1);
                } else if (lengthOfFileName > 12) {
                    //According to the s3g spec, only 12-character FN's supported
                    redirectToMessage(&socket, "Filename too long", 1);
                } else if (memcmp(filename+lengthOfFileName-4, ".X3G", 4)==0 ||
                        memcmp(filename+lengthOfFileName-4, ".x3g", 4)==0) {                             
                        //Send command to bot to print
                        if (bot) {
                            bot->playbackCapture(filename);
                            redirectToMessage(&socket, "Beginning build", 0); 
                        } else {                                   
                            redirectToMessage(&socket, "Error: connection to Makerbot not established.", 1); 
                        }
                } else {
                    redirectToMessage(&socket, "Error: filename doesn't end with X3G.", 1);
                }                
            } else {                    
                redirectToMessage(&socket, "Unknown command", 1);
            }
        } else {
            sendHTTPHeader(&socket, 404, "Not Found", "text/html"); 
            sendPageHeader(&socket);
            socket.printf("404 Not Found");
            sendPageFooter(&socket);  
        }        
           
        //DISPLAY("Closing socket\r\n");
        socket.close();
        free(path);
    }
    
    // never gets here
    //EthernetInterface::disconnect();  
}

int findOccuranceOfChar(char* buffer, char c, int occur, int max_length) {
    int cnt = 0;
    for (int x=0; x<max_length; x++) {
        if (buffer[x] == c) {
            cnt++;
            if (cnt == occur) {
                return x;
            }
        }
    }
    return -1;
}

void sendPageHeader(ClientSocket* sock) {
    sock->printf("<html>");
    sock->printf("<head><style>");
    sock->printf("body {font-family:Helvetica, Arial, sans-serif}");
    sock->printf("table { border:1px solid #000; border-collapse:collapse;}");
    sock->printf("td {border: 1px solid}");
    sock->printf("tr { vertical-align:middle; }");
    sock->printf("td.icon { text-align:center; padding:5px; }");
    sock->printf("a {color: #000000; text-decoration: none; }");
    sock->printf("</style>");
    sock->printf("<body>");
}

void sendStatusContent(ClientSocket* sock) {
    if (bot) {
        Makerbot::MakerbotBuildState state = bot->getBuildState();
        sock->printf("<H1>Build Status</H1>");
        sock->printf("Machine Name: %s<br>", bot->getMachineName());
        sock->printf("Bot firmware version: %0.1f<br><br>", bot->getMakerbotVersion());        
        sock->printf("<table width=350>");
        sock->printf("<tr><td>Build state</td><td>%s</td></tr>", Makerbot::humanReadableBuildState(state));
        sock->printf("<tr><Td>Build file name</td><td>%s</td></tr>", bot->getBuildName());
        sock->printf("<tr><td>Elapsed Build Time</td><td>%d:%02d</td></tr>", state.hours, state.minutes);
        sock->printf("<tr><td>Line Number</td><td>%d</td></tr>", state.line);
        sock->printf("</table>");
        
        sock->printf("<H1>Temperatures</H1>");
        sock->printf("<table width=350>");
        for (int x=0; x<bot->getToolCount(); x++) {                            
            sock->printf("<tr><td>Tool %d</td><td>%d&#176;C</td><td>%d&#176;C</td></tr>", x,  bot->getToolTemperature(x), bot->getToolSetPoint(x));
        }            
        if (bot->hasPlatform()) {
            sock->printf("<tr><td>Platform</td><td>%d&#176;C</td><td>%d&#176;C</td></tr>", bot->getPlatformTemperature(0), bot->getPlatformSetPoint(0));
        }
        sock->printf("</table>");        
    } else {
        sock->printf("<H1>No connection to Makerbot</H1>");
    }
}

void sendHTTPHeader(ClientSocket* sock, int response_code, char* response_descr, char* contentType) {
    sock->printf("HTTP/1.1 %d %s\r\n", response_code, response_descr);
    if (contentType != NULL)
        sock->printf("Content-Type: %s\r\n", contentType);
    sock->printf("\r\n");
}

void sendHTTPRedirect(ClientSocket* sock, char* redirectTo) {
    sock->printf("HTTP/1.1 %d %s\r\n", 302, "Found");
    sock->printf("Location: %s\r\n", redirectTo);
    sock->printf("\r\n");
}

void redirectToMessage(ClientSocket* sock, char* message, int isErr) {
    sock->printf("HTTP/1.1 %d %s\r\n", 302, "Found");
    char* escapedMsg = url_encode(message);
    sock->printf("Location: /?%s=%s\r\n", isErr?"err":"msg", escapedMsg);
    free(escapedMsg);
    sock->printf("\r\n");    
}

void renderMessageBox(ClientSocket* sock, char* message, int isBadMessage) {
    if (isBadMessage)
        sock->printf("<table><tr><td style='padding: 15px; background: #FF9999'>");
    else
        sock->printf("<table><tr><td style='padding: 15px; background: #99FF99'>");
    sock->printf("%s", message);
    sock->printf("</td><tr></table>");
    
}


void sendCommandTable(ClientSocket* sock) {
    if (bot) {
        Makerbot::MakerbotBuildState state = bot->getBuildState();
        sock->printf("<H1>Commands</H1>");
        sock->printf("<table width=500>");
        if (state.build_state == Makerbot::BUILD_RUNNING || 
            state.build_state == Makerbot::BUILD_PAUSED) {        
            sock->printf("<tr><td class=icon><a href='/command?cancel'>&#9608</a></td><td><a href='/command?cancel'>Cancel</a></td><td>Halt build immediately (does not clear build plate)</td></tr>");
        }
        if (state.build_state == Makerbot::BUILD_RUNNING) {
            sock->printf("<tr><td class=icon><a href='/command?pause'>&#x275a; &#x275a;</a></td><td><a hred='/command?pause'>Pause</a></td><td>Pause build and clear build area</td></tr>");
        }
        if (state.build_state == Makerbot::BUILD_PAUSED) {
            sock->printf("<tr><td class=icon><a href='/command?resume'>&#9658</a></td><td><a href='/command?resume'>Resume</a></td><td>Resume from pause</td></tr>");
        }
        if (state.build_state == Makerbot::NO_BUILD || 
            state.build_state == Makerbot::BUILD_FINISHED_NORMALLY ||
            state.build_state == Makerbot::BUILD_CANCELLED ||
            state.build_state == Makerbot::BUILD_SLEEPING) {
            sock->printf("<tr><td class=icon>&#9658</td><td>Play</TD><TD><form action='/command' method=get><input length=12 maxlength=12 name=build><input type=submit></form></td></tr>");
        }
        sock->printf("</table>");
    }
}

void sendPageFooter(ClientSocket* sock) {
    sock->printf("<br><hr><br>Server firmware version: %s<br>", FIRMWARE_VERSION);   
    time_t ctTime = time(NULL);
    sock->printf("Page generated at %s\r\n", ctime(&ctTime));                         
    sock->printf("</body>");
    sock->printf("</html>");
}

// Public domain code from http://www.geekhideout.com/urlcode.shtml
// To the extent possible under law, Fred Bulback has waived all copyright and related or neighboring rights to URL Encoding/Decoding in C. This work is published from: United States.
bool isalnum(char c) {
    if (c >= 48 && c <= 57) return 1;  //numbers
    if (c >= 65 && c <= 90) return 1;  //upercase
    if (c >= 97 && c <= 122) return 1;  //lowercase
    return 0;
}

bool isdigit(char c) {
    if (c >= 48 && c <= 57) return 1;  //numbers
    return 0;
}

char tolower(char c) {
    if (c >= 65 && c <= 90) {
        return c+32;
    } else {
        return c;
    }
}


/* Converts a hex character to its integer value */
char from_hex(char ch) {
  return isdigit(ch) ? ch - '0' : tolower(ch) - 'a' + 10;
}

/* Converts an integer value to its hex character*/
char to_hex(char code) {
  static char hex[] = "0123456789abcdef";
  return hex[code & 15];
}

/* Returns a url-encoded version of str */
/* IMPORTANT: be sure to free() the returned string after use */
char *url_encode(char *str) {
  char *pstr = str, *buf = (char*)malloc(strlen(str) * 3 + 1), *pbuf = buf;
  while (*pstr) {
    if (isalnum(*pstr) || *pstr == '-' || *pstr == '_' || *pstr == '.' || *pstr == '~') 
      *pbuf++ = *pstr;
    else if (*pstr == ' ') 
      *pbuf++ = '+';
    else 
      *pbuf++ = '%', *pbuf++ = to_hex(*pstr >> 4), *pbuf++ = to_hex(*pstr & 15);
    pstr++;
  }
  *pbuf = '\0';
  return buf;
}

/* Returns a url-decoded version of str */
/* IMPORTANT: be sure to free() the returned string after use */
char *url_decode(char *str) {
  char *pstr = str, *buf = (char*)malloc(strlen(str) + 1), *pbuf = buf;
  while (*pstr) {
    if (*pstr == '%') {
      if (pstr[1] && pstr[2]) {
        *pbuf++ = from_hex(pstr[1]) << 4 | from_hex(pstr[2]);
        pstr += 2;
      }
    } else if (*pstr == '+') { 
      *pbuf++ = ' ';
    } else {
      *pbuf++ = *pstr;
    }
    pstr++;
  }
  *pbuf = '\0';
  return buf;
}

// Search through the HTTP headers and  if there is an authentication line, process it
// Discard headers that are not relevant.
int handleAuthorizationHeaders(ClientSocket* socket, char* desiredToken) {
    char line [HTTP_LINE_BUFFER_SIZE];    
    int length=0;
    char* authToken = NULL;
    while ((length = readline(socket, line, HTTP_LINE_BUFFER_SIZE)) > 0) {
        if (memcmp(line, "Authorization: Basic ", 21) == 0) {
            //we found the authorization line.
            authToken = (char*)malloc(length - 21 + 1);
            memcpy(authToken, line+21, length-21); 
            authToken[length-21] = 0; //terminator            
            break;
        } else {
            //nothing
        }
        Thread::yield();
    }
    
    if (authToken != NULL) {         
        int match = memcmp(desiredToken, authToken, strlen(authToken)); 
        free(authToken);
        if (match == 0) return 1;
    }
    
    return 0;
}

// Discard line by line until a blank line 
// (the separator between the header and the body of the HTTP request)
void discardHeaders(ClientSocket* socket) {   
    if (!socket->available()) {
        //printf("nothing available, so not doing anything about the headers\r\n");
        return;
    }
    
    char line [HTTP_LINE_BUFFER_SIZE];
    while (readline(socket, line, HTTP_LINE_BUFFER_SIZE) > 0) {        
        Thread::yield();
    }
}

// Return the base64 representation of the input
// IMPORTANT: caller must call free() on the result when done with it.
char* base64(char* input) {
    int inputLen = strlen(input);
    int padLength = (3-(inputLen % 3))%3;
    
    int outNdx=0;
    char* output = (char*)malloc(((4*inputLen)/3)+1);
    
    for (int i=0; i < inputLen; i+=3) {
        //Assemble the 24 bit number
        uint32_t a = input[i] << 16;
        if (i+1 < inputLen) a += input[i+1] << 8;
        if (i+2 < inputLen) a += input[i+2];
        
        //Split the 24 bit number into 4 parts
        uint8_t p1 = (uint8_t)(a >> 18) & 0x3F;
        uint8_t p2 = (uint8_t)(a >> 12) & 0x3F;
        uint8_t p3 = (uint8_t)(a >> 6)  & 0x3F;
        uint8_t p4 = (uint8_t)a & 0x3F;
    
        //write the data to the output
        output[outNdx++] = base64table[p1];
        output[outNdx++] = base64table[p2];
        output[outNdx++] = base64table[p3];
        output[outNdx++] = base64table[p4]; 
    }
        
    // Put the padding in at the end    
    for (int i=0; i<padLength; i++) {
        output[outNdx-i-1]='=';
    }
    
    // Terminating \0
    output[outNdx]='\0';

    return output;      
}
