/**************************************************************************************************
 *****                                                                                        *****
 *****  Name: SPI_TFT.h                                                                       *****
 *****  Ver.: 1.0                                                                             *****
 *****  Date: 04/01/2013                                                                      *****
 *****  Auth: Frank Vannieuwkerke                                                             *****
 *****        Erik Olieman                                                                    *****
 *****  Func: library for 240*320 pixel TFT with ILI9320 LCD Controller                       *****
 *****                                                                                        *****
 *****  Rewrite from Peter Drescher code - http://mbed.org/cookbook/SPI-driven-QVGA-TFT       *****
 *****                                                                                        *****
 **************************************************************************************************/

#ifndef MBED_SPI_TFT_H
#define MBED_SPI_TFT_H

/* This library allows you to store a background image on the local flash memory of the microcontroller, 
from any filesystem (such as SD cards). This allows very fast writing of this specific image, and it allows
you to write text in a nice way over the image. However it does cost the last 5 flash sectors of the LPC1768.
Generally that won't be a problem, if it is a problem, add #define NO_FLASH_BUFFER before including this file.
*/


#include "GraphicsDisplay.h"
#include "BurstSPI.h"
#include "mbed.h"

#ifndef NO_FLASH_BUFFER
    #include "IAP.h"
#endif


#define incx() x++, dxt += d2xt, t += dxt
#define incy() y--, dyt += d2yt, t += dyt

#define SPI_F_LO    10000000
#define SPI_F_HI    48000000

/* some RGB565 color definitions                                                 */
#define Black           0x0000      /*   0,   0,   0 */
#define Navy            0x000F      /*   0,   0, 128 */
#define DarkGreen       0x03E0      /*   0, 128,   0 */
#define DarkCyan        0x03EF      /*   0, 128, 128 */
#define Maroon          0x7800      /* 128,   0,   0 */
#define Purple          0x780F      /* 128,   0, 128 */
#define Olive           0x7BE0      /* 128, 128,   0 */
#define LightGrey       0xC618      /* 192, 192, 192 */
#define DarkGrey        0x7BEF      /* 128, 128, 128 */
#define Blue            0x001F      /*   0,   0, 255 */
#define Green           0x07E0      /*   0, 255,   0 */
#define Cyan            0x07FF      /*   0, 255, 255 */
#define Red             0xF800      /* 255,   0,   0 */
#define Magenta         0xF81F      /* 255,   0, 255 */
#define Yellow          0xFFE0      /* 255, 255,   0 */
#define White           0xFFFF      /* 255, 255, 255 */
#define Orange          0xFD20      /* 255, 165,   0 */
#define GreenYellow     0xAFE5      /* 173, 255,  47 */

/** Class to use TFT displays with an ILI9320 controller using SPI mode
*/
 class SPI_TFT : public GraphicsDisplay {
 public:

  /** Create a SPI_TFT object connected to SPI and two pins
   *
   * @param mosi,miso,sclk SPI
   * @param cs pin connected to CS of display
   * @param reset pin connected to RESET of display
   *
   */
  SPI_TFT(PinName mosi, PinName miso, PinName sclk, PinName cs, const char* name ="TFT");

  /** Write a value to the to a LCD register
   *
   * @param reg register to be written
   * @param val data to be written
   */
  void wr_reg (unsigned char reg, unsigned short val);

  /** Get the width of the screen in pixels
   *
   * @param
   * @returns width of screen in pixels
   *
   */
  virtual int width();

  /** Get the height of the screen in pixels
   *
   * @returns height of screen in pixels
   *
   */
  virtual int height();

  /** Draw a pixel at x,y with color
   *
   * @param x (horizontal position)
   * @param y (vertical position)
   * @param color (16 bit pixel color)
   */
  virtual void pixel(int x, int y,int colour);

  /** Set draw window region to whole screen
   *
   */
  void WindowMax (void);

  /** draw a 1 pixel line
   *
   * @param x0,y0 start point
   * @param x1,y1 stop point
   * @param color 16 bit color
   *
   */
  void line(int x0, int y0, int x1, int y1, int colour);

  /** draw a rect
   *
   * @param x0,y0 top left corner
   * @param w,h   width and height
   * @param color 16 bit color
   *                                                   *
   */
  void rect(int x0, int y0, int w, int h, int colour);

  /** draw a filled rect
   *
   * @param x0,y0 top left corner
   * @param w,h   width and height
   * @param color 16 bit color
   *
   */
  void fillrect(int x0, int y0, int w, int h, int colour);

  /** draw an ellipse - source : http://enchantia.com/graphapp/doc/tech/ellipses.html
   *
   * @param xc,yc center point
   * @param a,b semi-major axis and semi-minor axis
   * @param color 16 bit color
   *
   */
  void draw_ellipse(int xc, int yc, int a, int b, unsigned int color);

  /** draw a filled ellipse - source : http://enchantia.com/graphapp/doc/tech/ellipses.html
   *
   * @param xc,yc center point
   * @param a,b semi-major axis and semi-minor axis
   * @param color 16 bit color
   *
   */
  void fill_ellipse(int xc, int yc, int a, int b, unsigned int color);

  /** setup cursor position
   *
   * @param x x-position (top left)
   * @param y y-position
   */
  virtual void locate(int x, int y);

  /** Fill the screen with _background color
   *
   */
  virtual void cls (void);

  /** Read ILI9320 ID
   *
   * @returns LCD ID code
   *
   */
  unsigned short Read_ID(void);

  /** calculate the max number of char in a line
   *
   * @returns max columns
   * depends on actual font size
   *
   */
  virtual int columns(void);

  /** calculate the max number of rows
   *
   * @returns max rows
   * depends on actual font size
   *
   */
  virtual int rows(void);

  /** put a char on the screen
   *
   * @param value char to print
   * @returns printed char
   *
   */
  virtual int _putc(int value);

  /** draw a character on given position out of the active font to the TFT
   *
   * @param x x-position of char (top left)
   * @param y y-position
   * @param c char to print
   *
   */
  virtual void character(int x, int y, int c);

  /** paint a bitmap on the TFT
   *
   * @param x,y : upper left corner
   * @param w width of bitmap
   * @param h high of bitmap
   * @param *bitmap pointer to the bitmap data
   *
   *   bitmap format: 16 bit R5 G6 B5
   *
   *   use Gimp to create / load , save as BMP, option 16 bit R5 G6 B5
   *   use winhex to load this file and mark data stating at offset 0x46 to end
   *   use edit -> copy block -> C Source to export C array
   *   paste this array into your program
   *
   *   define the array as static const unsigned char to put it into flash memory
   *   cast the pointer to (unsigned char *) :
   *   tft.Bitmap(10,40,309,50,(unsigned char *)scala);
   */
  void Bitmap(unsigned int x, unsigned int y, unsigned int w, unsigned int h, unsigned char *Name_BMP);


   /** paint a BMP (16/24 bit) from filesytem on the TFT (slow)
   *
   * The 16-bit format is RGB-565. Using 16-bit is faster than 24-bit, however every program can
   * output 24 bit BMPs (including MS paint), while it is hard to get them in the correct 16-bit format.
   *
   * @param x,y : upper left corner
   * @param *Name_BMP location of the BMP file (for example "/sd/test.bmp")
   * @returns 1 if bmp file was found and stored
   * @returns 0 if file was not found
   * @returns -1 if file was not a bmp
   * @returns -2 if bmp file is not 16/24bit
   * @returns -3 if bmp file is wrong for screen
   * @returns -4 if buffer malloc goes wrong
   *
   */

  int Bitmap(unsigned int x, unsigned int y, const char *Name_BMP);



  /** select the font to use
   *
   * @param f pointer to font array
   *
   *   font array can be created with GLCD Font Creator from http://www.mikroe.com
   *   you have to add 4 parameter at the beginning of the font array to use:
   *   - the number of bytes / char
   *   - the vertial size in pixels
   *   - the horizontal size in pixels
   *   - the number of bytes per vertical line
   *   you also have to change the array to char[]
   *
   */
  void set_font(unsigned char* f);

  /** Set the orientation of the screen
   *  x,y: 0,0 is always top left
   *
   * @param o direction to use the screen (0-3) 90&#65533; Steps
   *
   */
  void set_orientation(unsigned int o);


  /** Modify the orientation of the screen
   *  x,y: 0,0 is always top left
   *
   * @param none
   *
   * ILI9320 limitation: The orientation can only be used with a window command (registers 0x50..0x53)
   * 
   * reg 03h (Entry Mode) : BGR = 1 - ORG = 1 - ID0, ID1 and AM are set according to the orientation variable.
   * IMPORTANT : when ORG = 1, the GRAM writing direction follows the orientation (ID0, ID1, AM bits)
   *             AND we need to use the window command (reg 50h..53h) to write to an area on the display
   *             because we cannot change reg 20h and 21h to set the GRAM address (they both remain at 00h).
   *             This means that the pixel routine does not work when ORG = 1.
   *             Routines relying on the pixel routine first need to set reg 03h = 0x1030
   *             (cls, circle and line do so) AND need to write the data according to the orientation variable.
   */
  void mod_orientation(void);
  
  #ifndef NO_FLASH_BUFFER
  /** Move an image to the background buffer
  * 
  * The image must fit exactly on the screen (240x320). This function takes quite some time, depending on source filesystem.
  *
  * @param *Name_BMP location of the BMP file (for example "/sd/test.bmp")
  * @returns 1 if bmp file was found and stored
  * @returns 0 if file was not found
  * @returns -1 if file was not a bmp
  * @returns -2 if bmp file is not 16/24bit
  * @returns -3 if bmp file is wrong for screen
  * @returns -4 if buffer malloc goes wrong
  */
  int fileToFlash(const char *Name_BMP);
  
  /** Use the image on the flash memory as background
  *
  * @param active - true to use the image, false to use static color
  */
  void backgroundImage(bool active);
  #endif




  
// ------------------ PROTECTED PART ------------------
protected:

  /** draw a horizontal line
   *
   * @param x0 horizontal start
   * @param x1 horizontal stop
   * @param y vertical position
   * @param color 16 bit color
   *
   */
  void hline(int x0, int x1, int y, int colour);

  /** draw a vertical line
   *
   * @param x horizontal position
   * @param y0 vertical start
   * @param y1 vertical stop
   * @param color 16 bit color
   */
  void vline(int y0, int y1, int x, int colour);

  /** Set draw window region
   *
   * @param x horizontal position
   * @param y vertical position
   * @param w window width in pixel
   * @param h window height in pixels
   */
  virtual void window (int x, int y, int w, int h);

  /** Init the ILI9320 controller
   *
   */
  void tft_reset();

   /** Write data to the LCD controller
   *
   * @param dat data written to LCD controller
   *
   */
  void wr_dat(unsigned short value);

   /** Start data sequence to the LCD controller
   *
   */
  void wr_dat_start(void);

  /** Write a command the LCD controller
   *
   * @param cmd: command to be written
   *
   */
  void wr_cmd(unsigned char value);

  /** Read data from the LCD controller
   *
   * @returns data from LCD controller
   *
   */
  unsigned short rd_dat(void);

  /** Read a LCD register
   *
   * @param reg register to be read
   * @returns value of the register
   */
  unsigned short rd_reg (unsigned char reg);

  /** Set the cursor position
   *
   * @param x (horizontal position)
   * @param y (vertical position)
   *
   * Can only be used when reg 03h = 0x1030 (see note in mod_orientation).
   *
   */
  void SetCursor( unsigned short Xpos, unsigned short Ypos );
  
    struct bitmapData {
        int return_code;
        int width, height;
        int bits, pad;
        int start_data;
        FILE *file;
    };
    
  /** Get bitmap info
   *
   * @param   *Name_BMP  Bitmap filename
   * @returns structure: return_code        1 if bmp file was found and stored
   *                                        0 if file was not found
   *                                        -1 if file was not a bmp
   *                                        -2 if bmp file is not 16/24bit
   *                                        -3 if bmp file is wrong for screen
   *                     width, height      Bitmap size
   *                     bits, pad          BPP, padding (multiple of 4 bytes)
   *                     start_data         Starting address of the byte where the bitmap image data (pixel array) can be found
   *                     *file              Bitmap filename
   */
    bitmapData getBitmapData(const char *Name_BMP);

  unsigned int orientation;
  unsigned int char_x;
  unsigned int char_y;
  bool backgroundimage;
  BurstSPI _spi;
  DigitalOut _cs;
  unsigned char* font;
  #ifndef NO_FLASH_BUFFER
  IAP     iap;
  int backgroundOrientation;
  #endif

};

#endif
