/*
Copyright (c) 2018 Jacobus L de Jager

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/

#ifndef TMPx75_h
#define TMPx75_h

#include "mbed.h"

/* 
Library for TMP175 and TMP75 Temperature Sensors
For reference, read the datasheet: http://www.ti.com/lit/ds/symlink/tmp75.pdf

I2C ADDR (TMP75): 1 0 0 1 A2 A1 A0 (1001000 = 0x48)
*/

#define TEMPERATURE_REGISTER    0x00
#define CONFIGURATION_REGISTER  0x01
#define T_LOW_REGISTER          0x02
#define T_HIGH_REGISTER         0x03

//COnfiguration Register
//Register Byte: OS R1 R0 F1 F0 POL TM SD

//Default: continuous, comparator, high to low, 1 fault, 9 bits
#define DEFAULT_CONFIG          0x00

//Shutdown Mode (SD)
#define CONTINUOUS_MODE         0x00 
#define SHUTDOWN_MODE           0x01
 
//Thermostat Mode (TM)
#define COMPARATOR_MODE         0x00
#define INTERRUPT_MODE          0x02

//Polarity (POL)
#define HIGH_TO_LOW_POL         0x00
#define LOW_TO_HIGH_POL         0x04

//Fault Queue (F1/F0)
#define CONSECUTIVE_FAULTS_1    0x00
#define CONSECUTIVE_FAULTS_2    0x08
#define CONSECUTIVE_FAULTS_4    0x10
#define CONSECUTIVE_FAULTS_6    0x18

//Converter Resolution (R1/R0)
#define RESOLUTION_9_BITS       0x00    //  (0.5°C)     27.5 ms
#define RESOLUTION_10_BITS      0x20    //  (0.25°C)    55 ms
#define RESOLUTION_11_BITS      0x40    //  (0.125°C)   110 ms
#define RESOLUTION_12_BITS      0x60    //  (0.0625°C)  220 ms

//One-Shot (OS)
#define ONE_SHOT_MODE           0x80        


class TMPx75
{
public:
    TMPx75(I2C *i2c, uint8_t i2c_addr);
    virtual     ~TMPx75();
    Serial      pc;

    float       read_temperature();
    int         read_configuration();
    void        write_configuration(uint8_t config_byte);
    float       read_T_LOW();
    void        write_T_LOW(float temp);
    float       read_T_HIGH();
    void        write_T_HIGH(float temp);

private:
    I2C *_i2c;
    uint8_t _i2c_addr;
};

#endif

/*
// Example program for reading and writing to the TMPx75 temperature sensor
// Jacobus L de Jager 2018

#include "mbed.h"
#include "TMPx75.h"

Serial pc(SERIAL_TX, SERIAL_RX, 115200);
I2C i2c(I2C_SDA,I2C_SCL);
TMPx75 tmp75(&i2c, 0x48);

int main()
{
    printf("\nTMPx75 example\n");
    
    while(1) 
    {
        pc.printf("\nreading temperature\n");
        float temp = tmp75.read_temperature();
        printf("temp = %f\n", temp);
        
        pc.printf("writing config\n");
        tmp75.write_configuration(RESOLUTION_12_BITS);  
        
        pc.printf("reading config\n");
        int config = tmp75.read_configuration();
        printf("config register = 0x%x\n", config);

        pc.printf("reading T LOW\n");
        float T_LOW = tmp75.read_T_LOW();
        printf("T LOW = %f\n", temp);
        
        pc.printf("reading T HIGH\n");
        float T_HIGH = tmp75.read_T_HIGH();
        printf("T HIGH = %f\n", temp);
        
        pc.printf("writing T LOW\n");
        tmp75.write_T_LOW(25);
        
        pc.printf("writing T HIGH\n");
        tmp75.write_T_HIGH(50);

        printf("done!\n");
        
        wait(0.5);         
    }
}

*/
