/* University of York Robotics Laboratory Robot Arm Controller Board
 * 
 * Robot Arm Controller 
 *
 * File: robotarm.h
 *
 * (C) Dept. Electronics & Computer Science, University of York
 * 
 * James Hilder, Alan Millard, Shuhei Miyashita, Homero Elizondo, Jon Timmis
 *
 * February 2017, Version 1.0
 */ 


#ifndef ROBOTARM_H
#define ROBOTARM_H

#define SOFTWARE_VERSION_CODE 0.21
#define LCD_ADDRESS 0x7C

// Remote control definitions:
// REMOTE_ENABLED = 1 to enable remote control polling
// REMOTE_POLL_RATE = poll rate in microseconds
// REMOTE_LINEAR_STEPS = 1 to use same step amount each step; 0 to use a exponentially increasing amount as button held
// REMOTE_START_SPEED = Size of step to use; suggest 2 for LINEAR_STEPS=0 or 10 for LINEAR_STEPS=1
// REMOTE_USE_CURRENT_POSITION = 0 will move based on the stored target position, 1 will read the current position and set target accordingly.  This may make certain settings combinations fail to move under cases of high-torque load and low step size
#define REMOTE_ENABLED 1
#define REMOTE_POLL_RATE 15627
#define REMOTE_LINEAR_STEPS 1
#define REMOTE_START_SPEED 10
#define REMOTE_USE_CURRENT_POSITION 0

#define HAS_BASE 1
#define HAS_SHOULDER 1
#define HAS_ELBOW 1
#define HAS_WRIST 0

#define INVERT_BASE 1
#define INVERT_SHOULDER 1
#define INVERT_ELBOW 0
#define INVERT_WRIST 0

// If USE_SOFT_LIMITS = 1, SERVOS WILL BE LIMITED TO RANGES THAT FOLLOW
#define USE_SOFT_LIMITS 1
#define BASE_LIMIT_LOW 100
#define BASE_LIMIT_HIGH 4000
#define SHOULDER_LIMIT_LOW 700
#define SHOULDER_LIMIT_HIGH 3400
#define ELBOW_LIMIT_LOW 380
#define ELBOW_LIMIT_HIGH 3720
#define WRIST_LIMIT_LOW 300
#define WRIST_LIMIT_HIGH 700
#define USE_LIMIT_WARNING 1

#define BASE 10
#define SHOULDER 11
#define ELBOW 12
#define WRIST 13

#include "mbed.h"
#include "display.h"
#include "remote.h"
#include "servo.h"
#include "SerialHalfDuplex.h"

extern Display display;
extern Remote remote;
extern Serial pc;
extern Serial servo_data;
extern Servo servo;

extern char remote_on;
extern short target_base;
extern short target_shoulder;
extern short target_elbow;
extern short target_wrist;

/** Robotarm Class
 * The main class to define the robot arm
 *
 * Example code for main.cpp:
 * @code
 * #include "robotarm.h"
 * Robotarm arm;
 * int main(){
 *    arm.init();
 *    while(1) { //Do something!
 *    }
 * }
 * @endcode
 */
class Robotarm
{
public:
    /**
     * Main initialisation routine for the robot arm
     *
     * Set up the display, set up listener for remote control, set up servos
     */
    void init(void);
    
    /**
     * Test and initialise the servo motors
     */
    void init_servos(void);

    /**
     * Zero the servo motors (set to center position and activate)
     * @param time_delay: The time delay in seconds to wait before zeroing (0=no delay)
     */
    void zero_servos(float time_delay);
    
    /**
     * Called when servo initialisation fails; displays warning and halts program.
     */
    void fail_init(void);
    
    /**
     * Hardware test scans for all servos at different baud rates
     */
    void hardware_test(void); 

    /**
     * Initialise_servo is used to set the EPROM on a single servo so that its
     * ID and baud rate are correct; only to be used in initial configuration of the
     * arm, not in a normal program
     */
    void initialise_servo(int target_servo);
    
    private:
    
    void _zero_servos(void);
    void _zero_servos_display_update(void);
};
#endif // ROBOTARM_H

/*
 * Copyright 2017 University of York
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on an "AS IS" BASIS
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and limitations under the License.
 *
 */