/* University of York Robotics Laboratory Robot Arm Controller Board
 *
 * Copyright 2017 University of York
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on an "AS IS" BASIS
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and limitations under the License.
 *
 * File: display.cpp
 *
 * (C) Dept. Electronics & Computer Science, University of York
 *
 * James Hilder, Alan Millard, Shuhei Miyashita, Homero Elizondo, Jon Timmis
 *
 *
 * January 2017
 *
 * Driver for the Midas 16x2 I2C LCD Display (MCCOG21605x6W) LCD
 * [Farnell part 2218942 or 2063206]
 *
 */

#include "robotarm.h"


Timeout init_timeout;


Display::Display(PinName sda, PinName scl) :  Stream("display"), _i2c(sda,scl)
{
}

Display::Display() :  Stream("display"), _i2c(p28,p27)
{
}

int Display::i2c_message(char byte)
{
    char bytes [2];
    bytes[0]=0x80;
    bytes[1]=byte;
    int ret=_i2c.write(LCD_ADDRESS,bytes,2);
    wait(0.01);
    return ret;
}

int Display::disp_putc(int c)
{
    char message [2];
    message[0]=0x40;
    message[1]=c;
    _i2c.write(LCD_ADDRESS,message,2);
    wait(0.01);
    return c;
}

void Display::init()
{
    //Set initial states: display on, cursor off
    display_on = 1;
    cursor_on = 0;
    blink_on  = 0;

    i2c_message(0x38);
    i2c_message(0x39);
    i2c_message(0x14);
    i2c_message(0x74);
    i2c_message(0x54);
    i2c_message(0x6F);
    _set_display();
    clear_display();
    wait(0.05);
    clear_display();
    set_position(0,0);
    write_string("  YORK ROBOTICS");
    set_position(1,0);
    write_string("   LABORATORY");
    init_timeout.attach(this,&Display::post_init,0.3);
    wait(0.62);
}

void Display::post_init()
{
    clear_display();
    home();
    write_string("ROBOTIC ARM");
    set_position(1,0);
    char line [17];
    sprintf(line,"VERSION %1.2f", SOFTWARE_VERSION_CODE  );
    set_position(1,0);
    write_string(line);
    init_timeout.attach(this,&Display::post_post_init,0.3);
}

void Display::post_post_init()
{
    clear_display();
    home();
}

void Display::write_string(char * message)
{
    size_t length = strlen(message);
    if (length > 16) length = 16;
    char to_send [length+1];
    to_send[0]=0x40;
    for(int i=0; i<length; i++) {
        to_send[i+1] = message[i];
    }
    _i2c.write(LCD_ADDRESS,to_send,length+1);
   
}


void Display::write_string(char * message, char length)
{
    char to_send [length+1];
    to_send[0]=0x40;
    for(int i=0; i<length; i++) {
        to_send[i+1] = message[i];
    }
    _i2c.write(LCD_ADDRESS,to_send,length+1);
  
}

void Display::set_position(char row, char column)
{
    if(row < 2 && column < 16) {
        char pos = 128 +((row * 64)+column);
        i2c_message(pos);
    }
}

void Display::set_cursor(char enable)
{
    cursor_on=enable;
    _set_display();
}

void Display::set_blink(char enable)
{
    blink_on=enable;
    _set_display();
}

void Display::set_display(char enable)
{
    display_on=enable;
    _set_display();
}


void Display::clear_display()
{
    i2c_message(0x01);
}

void Display::home()
{
    i2c_message(0x02);
}


void Display::_set_display()
{
    char mode = 8;
    if(display_on>0) mode += 4;
    if(cursor_on>0) mode += 2;
    if(blink_on>0) mode ++;
    i2c_message(mode);
}


int Display::_putc (int c)
{
    putc(c);
    return(c);
}

int Display::_getc (void)
{
    char r = 0;
    return(r);
}

/*
 * Copyright 2017 University of York
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on an "AS IS" BASIS
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and limitations under the License.
 */
