#include "mbed.h"
#include "C12832.h"
#include "SMARTGPU2.h"
#include "HallEffect.h"
#include "TimerCB.h"

AnalogIn Ain(p17);
AnalogOut Aout(p18);
Ticker s20khz_tick;
Ticker s1hz_tick;
Ticker s30hz_tick;
C12832 lcdX(p5, p7, p6, p8, p11);
SMARTGPU2 lcd(TXPIN, RXPIN, RESETPIN);
InterruptIn interrupt(p9);

//DigitalIn joyDown(p12);
//DigitalIn joyRight(p16);
InterruptIn startStop(p12); //joystick down
InterruptIn reset(p16); //joystick right

void s20khz_task(void);
void s1hz_task(void);
void screenRefresh(void);
void hallISR(void);
void startStopTimer(void);
void resetTimer(void);

float data_in, data_out;
HallEffect hall;
TimerCB timerCB;

//cache data for fixing LCD flicker
int lastCount;
int lastSPM;
int lastMin;
int lastSec;
int lastMS;

void lcdInit() {
    lcd.reset();
    lcd.start();
    lcd.erase();
    lastCount = -1;
    lastSPM = -1;
    lastMin = -1;
    lastSec = -1;
    lastMS = -1;
}

void drawOut()  {
    NUMBEROFBYTES bytes;
    //config strings
    lcd.setTextSize(FONT8);
    //high baud rate actually slows the system down
//    lcd.baudChange(BAUD7);             //set a fast baud! for fast drawing
    
    int spm = hall.getSPM();
    if(spm != lastSPM){
        lastSPM = spm;
        //draw black box over where rating is to clear it for redrawing
        lcd.drawRectangle(50,50,240,120,BLACK,FILL);
        //stroke rating using snprintf
        char spmBuffer[3];
        snprintf(spmBuffer, 3, "%02d", spm);
        lcd.string(50,50,MAX_X_LANDSCAPE,MAX_Y_LANDSCAPE,spmBuffer,&bytes);
    }
    
    int count = hall.getCount();
    if(count != lastCount){
        lastCount = count;
        //draw black box over where count is to clear it for redrawing
        lcd.drawRectangle(300,50,MAX_X_LANDSCAPE,120,BLACK,FILL);
        //stroke count using snprintf
        char strokesBuffer[4];
        snprintf(strokesBuffer, 4, "%03d", count);
        lcd.string(300,50,MAX_X_LANDSCAPE,MAX_Y_LANDSCAPE,strokesBuffer,&bytes);
    }
    
    /*
        TODO: make intelligent to only update (draw box):
        minutes if minutes changed
        seconds if sectonds changed
        ms if ms changed
        
        Adapt to number of ones present in higher places
    */
    
    int tenthSec = timerCB.getMS()/100;
    if(tenthSec != lastMS){
        lastMS = tenthSec;
        //draw black box over where the time is to clear it for redrawing
        lcd.drawRectangle(50,200,MAX_X_LANDSCAPE,MAX_Y_LANDSCAPE,BLACK,FILL);
        //time using snprintf
        char timeBuffer[8];
        snprintf(timeBuffer, 8, "%02d:%02d.%d",timerCB.getMinutes(),timerCB.getSeconds(),tenthSec);
        lcd.string(50,200,MAX_X_LANDSCAPE,MAX_Y_LANDSCAPE,timeBuffer,&bytes);
    }
    
    //print labels
    lcd.setTextSize(FONT4);
    lcd.string(50,170,MAX_X_LANDSCAPE,MAX_Y_LANDSCAPE,"Time",&bytes); 
    lcd.string(300,20,MAX_X_LANDSCAPE,MAX_Y_LANDSCAPE,"Count",&bytes);    
    lcd.string(50,20,MAX_X_LANDSCAPE,MAX_Y_LANDSCAPE,"SPM",&bytes);
}

int main(){
//    lcd.printf("Audio In");
    //initialize big LCD
    lcdInit();
    
    s20khz_tick.attach_us(&s20khz_task,50);
    //refresh app board lcd at .5hz
//    s1hz_tick.attach(&s1hz_task,.5);
    //refresh screen at 30hz
//    s30hz_tick.attach(&screenRefresh,.5);

    interrupt.rise(&hallISR);
    startStop.rise(&startStopTimer);
    reset.rise(&resetTimer);
    while(true){
      drawOut();
    }
}

void s20khz_task(void) {
    data_in = Ain;
    data_out = data_in;
    Aout = data_out;
}

void s1hz_task(void) {
    lcdX.cls();
    lcdX.locate(0,0);
    lcdX.printf("StrokeCount: %d\r\nStroke Rating: %d\n", hall.getCount(), hall.getSPM());
    lcdX.printf("time: %02d:%02d:%02d",timerCB.getMinutes(),timerCB.getSeconds(),(timerCB.getMS()/10));
}

void screenRefresh(void) {
    drawOut();
}

void hallISR() {
    hall.recordStroke();
    if(!timerCB.isRunning()){
        timerCB.start();
    }
}

void startStopTimer() {
    if(timerCB.isRunning()){
        timerCB.stop();
    }
    else {
        timerCB.start();
    }
}

void resetTimer() {
    timerCB.reset();
    hall.resetCount();
}