/**
 * @file ili9163lcd.h
 * @brief ILI9163 128x128 LCD Driver (Header file)
 *
 * This code has been ported from the ili9163lcd library for mbed
 * made by Jun Morita.
 * Source form <http://files.noccylabs.info/lib430/liblcd/ili9163lcd_8c.html>
 *
 * This code has been ported from the ili9163lcd library for avr made
 * by Simon Inns, to run on a msp430.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Jun Morita (iccraft)
 * @author Simon Inns <simon.inns@gmail.com>
 * @author Christopher Vagnetoft (NoccyLabs)
 * @copyright (C) 2012 Simon Inns
 * @copyright parts (C) 2012 NoccyLabs
 */

#ifndef ILI9163LCD_H_
#define ILI9163LCD_H_

#include "mbed.h"
#include "font5x8.h"

// Screen orientation defines:
// 0 = Ribbon at top
// 1 = Ribbon at left
// 2 = Ribbon at right
// 3 = Ribbon at bottom
#define LCD_ORIENTATION0    0
#define LCD_ORIENTATION1    96
#define LCD_ORIENTATION2    160
#define LCD_ORIENTATION3    192

// Each colour takes up 5 bits
// Green is shifted 6 bits to the left
// Blue is shifted 11 bits to the left
// some RGB color definitions      BLUE GREEN  RED                          
#define Black           0x0000      /*  0, 0, 0 */
#define Maroon          0x000F      /*  0, 0,15 */
#define DarkGreen       0x03C0      /*  0,15, 0 */
#define Navy            0x7800      /* 15, 0, 0 */
#define Red             0x001F      /*  0, 0,31 */
#define Green           0x07C0      /*  0,31, 0 */
#define Blue            0xF800      /* 31, 0, 0 */
#define Yellow          0x07DF      /*  0,31,31 */
#define Magenta         0xF81F      /* 31, 0,31 */
#define Cyan            0xFFC0      /* 31,31, 0 */
#define White           0xFFFF      /* 31,31,31 */

class ILI9163 {
  public:
    // Definitions for control lines (port C)
    static const uint8_t LCD_WR    =(1 << 2);
    static const uint8_t LCD_RS    =(1 << 4);
    static const uint8_t LCD_RD    =(1 << 5);
    static const uint8_t LCD_CS    =(1 << 6);
    static const uint8_t LCD_RESET =(1 << 7);
    // ILI9163 LCD Controller Commands
    static const uint8_t NOP                     = 0x00;
    static const uint8_t SOFT_RESET              = 0x01;
    static const uint8_t GET_RED_CHANNEL         = 0x06;
    static const uint8_t GET_GREEN_CHANNEL       = 0x07;
    static const uint8_t GET_BLUE_CHANNEL        = 0x08;
    static const uint8_t GET_PIXEL_FORMAT        = 0x0C;
    static const uint8_t GET_POWER_MODE          = 0x0A;
    static const uint8_t GET_ADDRESS_MODE        = 0x0B;
    static const uint8_t GET_DISPLAY_MODE        = 0x0D;
    static const uint8_t GET_SIGNAL_MODE         = 0x0E;
    static const uint8_t GET_DIAGNOSTIC_RESULT   = 0x0F;
    static const uint8_t ENTER_SLEEP_MODE        = 0x10;
    static const uint8_t EXIT_SLEEP_MODE         = 0x11;
    static const uint8_t ENTER_PARTIAL_MODE      = 0x12;
    static const uint8_t ENTER_NORMAL_MODE       = 0x13;
    static const uint8_t EXIT_INVERT_MODE        = 0x20;
    static const uint8_t ENTER_INVERT_MODE       = 0x21;
    static const uint8_t SET_GAMMA_CURVE         = 0x26;
    static const uint8_t SET_DISPLAY_OFF         = 0x28;
    static const uint8_t SET_DISPLAY_ON          = 0x29;
    static const uint8_t SET_COLUMN_ADDRESS      = 0x2A;
    static const uint8_t SET_PAGE_ADDRESS        = 0x2B;
    static const uint8_t WRITE_MEMORY_START      = 0x2C;
    static const uint8_t WRITE_LUT               = 0x2D;
    static const uint8_t READ_MEMORY_START       = 0x2E;
    static const uint8_t SET_PARTIAL_AREA        = 0x30;
    static const uint8_t SET_SCROLL_AREA         = 0x33;
    static const uint8_t SET_TEAR_OFF            = 0x34;
    static const uint8_t SET_TEAR_ON             = 0x35;
    static const uint8_t SET_ADDRESS_MODE        = 0x36;
    static const uint8_t SET_SCROLL_START        = 0x37;
    static const uint8_t EXIT_IDLE_MODE          = 0x38;
    static const uint8_t ENTER_IDLE_MODE         = 0x39;
    static const uint8_t SET_PIXEL_FORMAT        = 0x3A;
    static const uint8_t WRITE_MEMORY_CONTINUE   = 0x3C;
    static const uint8_t READ_MEMORY_CONTINUE    = 0x3E;
    static const uint8_t SET_TEAR_SCANLINE       = 0x44;
    static const uint8_t GET_SCANLINE            = 0x45;
    static const uint8_t READ_ID1                = 0xDA;
    static const uint8_t READ_ID2                = 0xDB;
    static const uint8_t READ_ID3                = 0xDC;
    static const uint8_t FRAME_RATE_CONTROL1     = 0xB1;
    static const uint8_t FRAME_RATE_CONTROL2     = 0xB2;
    static const uint8_t FRAME_RATE_CONTROL3     = 0xB3;
    static const uint8_t DISPLAY_INVERSION       = 0xB4;
    static const uint8_t SOURCE_DRIVER_DIRECTION = 0xB7;
    static const uint8_t GATE_DRIVER_DIRECTION   = 0xB8;
    static const uint8_t POWER_CONTROL1          = 0xC0;
    static const uint8_t POWER_CONTROL2          = 0xC1;
    static const uint8_t POWER_CONTROL3          = 0xC2;
    static const uint8_t POWER_CONTROL4          = 0xC3;
    static const uint8_t POWER_CONTROL5          = 0xC4;
    static const uint8_t VCOM_CONTROL1           = 0xC5;
    static const uint8_t VCOM_CONTROL2           = 0xC6;
    static const uint8_t VCOM_OFFSET_CONTROL     = 0xC7;
    static const uint8_t WRITE_ID4_VALUE         = 0xD3;
    static const uint8_t NV_MEMORY_FUNCTION1     = 0xD7;
    static const uint8_t NV_MEMORY_FUNCTION2     = 0xDE;
    static const uint8_t POSITIVE_GAMMA_CORRECT  = 0xE0;
    static const uint8_t NEGATIVE_GAMMA_CORRECT  = 0xE1;
    static const uint8_t GAM_R_SEL               = 0xF2;
    
    // Create the ILI9163 object
    //
    // @param D13 SCK
    // @param D11 SDA
    // @param D10 A0
    // @param D9  RESET    pin connected to RESET of display 
    // @param D8  CS       pin connected to CS of display 
    // @param D12 LED      (optional) connected to LED pin (for controlling backlight)
    //
    // ILI9163 lcd(D13,D11,D10,D9,D8,D12);
    //
    ILI9163(PinName SCK, PinName SDA, PinName A0, PinName RESET, PinName CS, PinName LED = NC);
    
    SPI SPI_;
    DigitalOut A0_;
    DigitalOut RESET_;
    DigitalOut CS_;
    DigitalOut LED_;
    
    // font array
    unsigned char* font;
    uint8_t font_bp_char;      // Bytes per character
    uint8_t font_hor;          // Horizontal size
    uint8_t font_vert;         // Vertical size
    uint8_t font_bp_line;      // Bytes per line
    
    //  LCD function prototypes
    void reset(void);
    void writeCommand(uint8_t address);
    void writeParameter(uint8_t parameter);
    void writeData(uint16_t data);
    void init(uint8_t orientation);
    
    //Theoretically, it is possible to use PWM to change the display brightness
    //However, I found that it can interfere with data transfers, so I just use backlight on/off
    inline void backlightOn() { LED_ = 1; };
    inline void backlightOff() { LED_ = 0; };
    inline void backlightToggle() { LED_ = !LED_; };
    
    // Translates a 3 byte RGB value into a 2 byte value for the LCD (values should be 0-31)
    static inline uint16_t colourFromRGB5(uint8_t r, uint8_t g, uint8_t b)
    { return (b << 11) | (g << 6) | (r); }
    
    static inline uint16_t colourFromRGB8(uint8_t r, uint8_t g, uint8_t b)
    { return colourFromRGB5(r >> 3, g >> 3, b >> 3); }
    
    void clearDisplay(uint16_t colour);
    void plot(uint8_t x, uint8_t y, uint16_t colour);
    void line(int16_t x0, int16_t y0, int16_t x1, int16_t y1, uint16_t colour);
    void rectangle(int16_t x0, int16_t y0, int16_t x1, int16_t y1, uint16_t colour);
    void filledRectangle(int16_t x0, int16_t y0, int16_t x1, int16_t y1, uint16_t colour);
    void circle(int16_t xCentre, int16_t yCentre, int16_t radius, uint16_t colour);
    
    // select the font to use
    //
    // @param f pointer to font array 
    //                                                                              
    //   font array can be created with GLCD Font Creator from http://www.mikroe.com
    //   you have to add 4 parameter at the beginning of the font array to use it: 
    //   - the number of bytes per char
    //   - the vertical size in pixel
    //   - the horizontal size in pixel
    //   - the number of bytes per vertical line
    //   you also have to change the array to char[]
    // 
    void setFont(unsigned char* f);
    inline uint8_t lcdTextX(uint8_t x) { return x*font_hor; }
    inline uint8_t lcdTextY(uint8_t y) { return y*font_vert; }
    
    void putCh(unsigned char character, uint8_t x, uint8_t y, uint16_t fgColour, uint16_t bgColour);
    void putS(const char *string, uint8_t x, uint8_t y, uint16_t fgColour, uint16_t bgColour);

};  // end class

#endif /* ILI9163LCD_H_ */
