#include "ADXL345_I2C.h"

 ADXL345_I2C accelerometer(I2C_SDA, I2C_SCL);
 Serial pc(USBTX, USBRX);
 
 class RangeMapper
 {
     /*
     RangeMapper convert data from input range to output range
     Math formular is y_output = ratio * x_input + offset;
     */
     protected: 
     double _ratio;
     double _offset;
     double _outrange;
     public:
     RangeMapper(double x_min, double x_max, double y_min, double y_max)
      {
          _ratio = (y_max - y_min) / (x_max - x_min);
          _offset = y_min - x_min * _ratio;
          _outrange = y_max - y_min; 
      };
     ~RangeMapper(){};
     double getOutput(double x_input)
        {return x_input * _ratio + _offset;};
     void zoom(float ratio)
        {_ratio *=  ratio;};
     void shift(float ratio)
        {_offset +=  _outrange * ratio ; } ;
      
 };

 
 class DataProcess  
 {
     protected: 
     // Statistics
     int64_t _summary; 
     double _squardsum; // summary (xi^2)
     int64_t _numbers;
     int64_t _max;
     int64_t _min;
     // integration, assume time interval is same 
     int64_t _itg1; //1 order integration
     int64_t _itg2; //2 order integration
     int64_t _itg3; //3 order integration
     // differentiation, assume time interval is same
     int64_t _preData;  
     int64_t _dif1; //1 order differentiation
     int64_t _dif2; //2 order differentiation
     public:
     DataProcess(): 
        _summary(0), _squardsum(0), _numbers(0), _max(0), _min(0),  
        _itg1(0), _itg2(0), _itg3(0), _preData(0), _dif1(0), _dif2(0) {};
     ~DataProcess() {}; 
     void putData(int64_t x) {
         int64_t temp;
         if (x > _max) _max = x;
         if (x < _min) _min = x;
         _summary += x;
         _squardsum += x*x;
         _numbers ++;
         _itg1 += x - _summary / _numbers ;
         _itg2 += _itg1;
         _itg3 += _itg2;
         temp = _dif1;
         _dif1 = x - _preData;
         _preData = x;
         _dif2 = _dif1 - temp; 
         };
     int64_t getSum(void) {return _summary;};
     double getMean(void) {return (double)_summary / (double)_numbers ;};
     double getStdDiv(void) {return sqrt((_squardsum - (double)_summary*_summary / (double)_numbers ) / (double)_numbers ); };
     int64_t getMax(void) {return _max; };
     int64_t getMin(void) {return _min; };
     int64_t getCount(void) {return _numbers;} ;
     int64_t getO1integration(void) {return _itg1;};
     int64_t getO2integration(void) {return _itg2;};
     int64_t getO3integration(void) {return _itg3;};
     int64_t GetO1differ(void) {return _dif1;};
     int64_t GetO2differ(void) {return _dif2;};
 };
 
 int main() {
     char getc =0;
     int channel = 0, datamode =0;
     bool output = true, mapout = true;
     pc.baud(115200);
     int readings[3] = {0, 0, 0};
     DataProcess dp1, dp2, dp3;
     DataProcess dp[3] = {dp1, dp2, dp3};
 
      
   
     pc.printf("Starting ADXL345 test...\r\n");
     wait(.001);
     pc.printf("Device ID is: 0x%02x\r\n", accelerometer.getDeviceID());
    wait(.001);
    
    restart:
    
     // These are here to test whether any of the initialization fails. It will print the failure
    if (accelerometer.setPowerControl(0x00)){
         pc.printf("didn't intitialize power control\r\n"); 
         return 0;  }
     wait(.001);
     
     //Full resolution, +/-16g, 4mg/LSB.
     if(accelerometer.setDataFormatControl(0x0B)){
     //Full resolution, +/-2g, 4mg/LSB.
     //if(accelerometer.setDataFormatControl(0x08)){
        pc.printf("didn't set data format\r\n");
        return 0;  }
     wait(.001);
     
     //3.2kHz data rate.
     if(accelerometer.setDataRate(ADXL345_3200HZ)){
        pc.printf("didn't set data rate\r\n");
        return 0;    }
     wait(.001);
     
     //Measurement mode.
     
     if(accelerometer.setPowerControl(MeasurementMode)) {
        pc.printf("didn't set the power control to measurement\r\n"); 
        return 0;   } 

     // pc.printf("x-axis, y-axis, z-axis\r\n"); 
     pc.printf("Data Commands: \r\n"); 
     pc.printf("  x -> x channdel \r\n"); 
     pc.printf("  y -> y channdel \r\n"); 
     pc.printf("  z -> z channdel \r\n"); 
     pc.printf("  o,0 -> data output on/off \r\n"); 
     pc.printf("  i,1 -> mapper on/off \r\n"); 
     pc.printf("  a -> raw data \r\n");
     pc.printf("  b -> mean \r\n");
     pc.printf("  c -> summary \r\n");
     pc.printf("  d -> variance \r\n");
     pc.printf("  e -> 1st order Integration \r\n");
     pc.printf("  f -> 2nd order Integration \r\n");
     pc.printf("  g -> 3rd order Integration \r\n");
     pc.printf("  h -> 1st order Differentiation \r\n");
     pc.printf("  k -> 2nd order Differentiation \r\n");
     pc.printf("  +,= -> Output Zoom In 2X \r\n");
     pc.printf("  - -> Output Zoom Out 2Z \r\n");
     pc.printf("  ^,6 -> shift up 1% \r\n");
     pc.printf("  v -> shift down 1% \r\n");
     pc.printf("Press any key to start. \r\n");
     getc = pc.getc();  
     
    int error_count=0;
    RangeMapper DAC_Mapper1( 0 - 1<<13, 1<<13, 0, 1);
    RangeMapper DAC_Mapper2( 0 - 1<<13, 1<<13, 0, 1);
 
     while (1) {
         if (pc.readable())
            {
                getc = pc.getc();
                switch( getc)
                    {
                        case 'x' : channel = 0 ; pc.printf("Data procesing output switch to X channdel \r\n"); break;
                        case 'y' : channel = 1 ; pc.printf("Data procesing output switch to Y channdel \r\n"); break;
                        case 'z' : channel = 2 ; pc.printf("Data procesing output switch to Z channdel \r\n"); break;
                        case '0' : 
                        case 'o' : output = !output ; pc.printf("Turn %s data output.\r\n",  (output ? "On" : "Off") ); break;
                        case '1' : 
                        case 'i' : mapout = !mapout ; pc.printf("Turn %s mapping output.\r\n",  (mapout ? "On" : "Off") ); break;
                        case 'a' : datamode = 0 ; pc.printf("Set to raw data output.\r\n"); break;
                        case 'b' : datamode = 1 ; pc.printf("Set to mean output.\r\n"); break;
                        case 'c' : datamode = 2 ; pc.printf("Set to summary output.\r\n"); break;
                        case 'd' : datamode = 3 ; pc.printf("Set to variance output.\r\n"); break;
                        case 'e' : datamode = 4 ; pc.printf("Set to 1st order Integration output.\r\n"); break;
                        case 'f' : datamode = 5 ; pc.printf("Set to 2nd order Integration output.\r\n"); break;
                        case 'g' : datamode = 6 ; pc.printf("Set to 3rd order Integration output.\r\n"); break;
                        case 'h' : datamode = 7 ; pc.printf("Set to 1st order Differentiation output.\r\n"); break;
                        case 'k' : datamode = 8 ; pc.printf("Set to 2nd order Differentiation output.\r\n"); break;
                        case '=' :
                        case '+' : DAC_Mapper2.zoom(2) ; pc.printf("Mapout zoom in.\r\n"); break;
                        case '-' : DAC_Mapper2.zoom(0.5) ; pc.printf("Mapout zoom out.\r\n"); break;
                        case '6' :
                        case '^' : DAC_Mapper2.shift(0.01) ; pc.printf("Mapout shift up.\r\n"); break;
                        case 'v' : DAC_Mapper2.shift(-0.01) ; pc.printf("Mapout shift down.\r\n"); break;
                        default: break;
                    }
            }
         accelerometer.getOutput(readings);
         if (( 17601 == readings[0] ) ||  ( 17601 == readings[1] ) || ( 17601 == readings[2] ))
            {
            error_count++;
            if (error_count>10)
                {
                accelerometer.setPowerControl(0);
                pc.printf("Sensor Halt!\r\n"); 
                goto restart;
                }
            }
        else
            {
            error_count = 0; 
            int i;
            for (i = 0; i<3; i++)
                dp[i].putData(readings[i] = (int16_t) readings[i] );
                
            if (output)
             {
               switch (datamode)
               {
                case 0: pc.printf("RAW: %i, %i, %i\r\n", (int16_t)readings[0], (int16_t)readings[1], (int16_t)readings[2]); break;
                //case 0: pc.printf("RAW: %i, %i, %i\r\n", readings[0], readings[1], readings[2]); break;
                case 1: pc.printf("MEAN: %f, %f, %f\r\n", dp[0].getMean(), dp[1].getMean(), dp[2].getMean() ); break;
                case 2: pc.printf("SUM: %jd, %jd, %jd\r\n", dp[0].getSum(), dp[1].getSum(), dp[2].getSum() ); break;                
                case 3: pc.printf("STD: %f, %f, %f\r\n", dp[0].getStdDiv(), dp[1].getStdDiv(), dp[2].getStdDiv() ); break;
                case 4: pc.printf("1ITG: %jd, %jd, %jd\r\n", dp[0].getO1integration(), dp[1].getO1integration(), dp[2].getO1integration() ); break;     
                case 5: pc.printf("2ITG: %jd, %jd, %jd\r\n", dp[0].getO2integration(), dp[1].getO2integration(), dp[2].getO2integration() ); break;     
                case 6: pc.printf("3ITG: %jd, %jd, %jd\r\n", dp[0].getO3integration(), dp[1].getO3integration(), dp[2].getO3integration() ); break;     
                case 7: pc.printf("1DIF: %jd, %jd, %jd\r\n", dp[0].GetO1differ(), dp[1].GetO1differ(), dp[2].GetO1differ() ); break;     
                case 8: pc.printf("2DIF: %jd, %jd, %jd\r\n", dp[0].GetO2differ(), dp[1].GetO2differ(), dp[2].GetO2differ() ); break;     
                default: break;         
               }
             }
             // Set data the DAC
             DAC_Mapper1.getOutput(readings[channel]);
             if (mapout)
             {
               double mapdata;
               switch (datamode)
               {
                case 0: mapdata = readings[channel];  break;
                case 1: mapdata = dp[channel].getMean(); break;
                case 2: mapdata = dp[channel].getSum();  break;                
                case 3: mapdata = dp[channel].getStdDiv();  break;
                case 4: mapdata = dp[channel].getO1integration(); break;     
                case 5: mapdata = dp[channel].getO2integration();  break;     
                case 6: mapdata = dp[channel].getO3integration();  break;     
                case 7: mapdata = dp[channel].GetO1differ(); break;     
                case 8: mapdata = dp[channel].GetO2differ(); break;     
                default: mapdata = 0; break;         
               }
               pc.printf("MAPPING OUTPUT: %f\r\n", DAC_Mapper2.getOutput(mapdata) );
               
             }
             else
             {
               DAC_Mapper2.getOutput(0);
             }

             wait(1);
            }

     }
 
 }
