/* mbed Microcontroller Library
 * Copyright (c) 2019 ARM Limited
 * SPDX-License-Identifier: Apache-2.0
 */

#include "mbed.h"
#include "platform/mbed_thread.h"
#include "LCD.h"
#include <string>


// Blinking rate in milliseconds
#define BLINKING_RATE_MS                                                    500

#define Init 0
#define Geschlossen 1
#define Oeffnen 2
#define Offen 3
#define Schliessen 4
#define Gestoppt 5
lcd mylcd;
DigitalIn TorOeffnen(PA_1);
DigitalIn TorSchliessen(PA_6);
InterruptIn Lichtschranke(PA_10);
DigitalIn EndschalterOffen(PB_0);
DigitalIn EndschalterZu(PB_3);
PwmOut heben(PC_6);
PwmOut senken(PC_7);
AnalogIn Geschwindigkeit(PA_0);
DigitalOut Warnleuchte(PC_0);

int zustand=Init;
float tastgrad;
string meldung[]={"Tor-offen","Tor-zu","Tor-gestoppt","Tor-faehrt"};

void blinkenEin()
{
    TIM6->CR1=1;
}

void blinkenAus()
{
    TIM6->CR1=0;
    Warnleuchte=0;
}

void isrTimer()
{
    TIM6->SR=0;
    Warnleuchte=!Warnleuchte;  
}

void initTimer()
{
   RCC->APB1ENR |= 0b10000; //TIM6
   TIM6->PSC=31999; //1ms 
   TIM6->ARR=499;   //500ms
   TIM6->CNT=0;     //Counter startet bei 0
   TIM6->SR=0;      //UIF =0
   TIM6->DIER=1;    //Interrupt Freigabe
   NVIC_SetVector(TIM6_IRQn,(uint32_t)&isrTimer);
   HAL_NVIC_EnableIRQ(TIM6_IRQn);
   __enable_irq();  
}

void oeffnen()
{
    senken=0;
    heben=Geschwindigkeit;
}

void schliessen()
{
    heben=0;
    senken=Geschwindigkeit;
}

void lcdOut(int nr)
{
  mylcd.clear();
  mylcd.cursorpos(0);
  mylcd.printf("%s",meldung[nr].c_str());   
}

void motorstop()
{
    heben=0;
    senken=0; 
}

void normaleFrequenz()
{
  TIM6->ARR=499;   //500ms  
}

void doppelteFrequenz()
{
   TIM6->CNT=0;     //Counter startet bei 0  
   TIM6->ARR=249;   //250ms
  
}
void schnellstop()
{
    if (zustand==Schliessen || zustand==Oeffnen)
    {
       zustand=Gestoppt;
       lcdOut(2); 
       doppelteFrequenz();
    }
}

void init()
{
    TorOeffnen.mode(PullDown);
    TorSchliessen.mode(PullDown);
    Lichtschranke.mode(PullDown);
    EndschalterOffen.mode(PullDown);
    EndschalterZu.mode(PullDown);
    Lichtschranke.fall(&schnellstop);
    Lichtschranke.enable_irq();
    __enable_irq();
    heben.period_us(100);
    senken.period_us(100);
    initTimer();
    
}


int main()
{
    while (true) {
     switch(zustand)
        {
        case Init: 
            init();
            zustand=Geschlossen;
            motorstop();
            lcdOut(1);
            break;
        case Geschlossen: if (TorOeffnen==1)
                {                 
                 zustand=Oeffnen;
                 blinkenEin();
                 lcdOut(3);
                }
                break;
        case Oeffnen: 
              oeffnen();
              if (EndschalterOffen==1)
              {
                  blinkenAus();
                  zustand=Offen;
                  motorstop();
                  lcdOut(0);
              }
              break;
        case Offen: if (TorSchliessen==1)
                {
                zustand=Schliessen;
                blinkenEin();
                lcdOut(3);
                }
                break;
        case Schliessen: 
                schliessen();
                if (EndschalterZu==1) 
                {
                    zustand=Geschlossen;
                    lcdOut(1);
                    motorstop();
                    blinkenAus();
                }
                break;
        case Gestoppt:
            motorstop();
            if (TorOeffnen==1)
                {
                 lcdOut(3);
                 zustand=Oeffnen;
                 blinkenEin();
                 normaleFrequenz();
                }
            if (TorSchliessen==1)
                {
                lcdOut(3);
                zustand=Schliessen;
                blinkenEin();
                normaleFrequenz();
                }
                break;
        }
        HAL_Delay(20);
    }
}
