// -*- coding: utf-8 -*-
/**
 @file      i2c_general_io.h
 @brief     This is a library for accessing registers of a typical i2c sensor to set or read measurement value.
 
 @author    D.Nakayama
 @version   1.0
 @date      2018-07-08  D.Nakayama  Written for C++/mbed.
 
 
 @see 
 Copyright (C) 2018 D.Nakayama.
 Released under the MIT license.
 http://opensource.org/licenses/mit-license.php
 
*/

#ifndef INCLUDED_i2c_genela_io_h_
#define INCLUDED_i2c_genela_io_h_
#include "mbed.h"

/**
    @class  GEN_I2C
    @brief  Class for accessing a typical sensor with i2c
*/
class GEN_I2C{

    public:
    
    /**
         @brief Create a new i2c sensor.
         @param sda   SDA pin name (Defined in PinName.h)
         @param sck   SCK pin name (Defined in PinName.h)
    */
    GEN_I2C(PinName sda, PinName sck);
    /**
         @brief Create a new i2c sensor.
         @param &i2c_obj   i2c name
    */
    GEN_I2C(I2C &i2c_obj);
    
    /**
         @brief Disable the i2c sensor.
         @param No parameters.
    */
    virtual ~GEN_I2C();


    //multi bytes func
    /**
        @brief multible read register.
        @param[in]  Device_add  Device i2c address(8bit)
        @param[in]  reg_add     register address(8bit)
        @param[out] *data       read data array
        @param[in]  n           Number of register reads
        @return int read result(define mbed(i2c))
    */
    virtual int read_reg(char Device_add, char reg_add, char *data, int n);
    
    //single byte func
    /**
        @brief single read register.
        @param[in]  Device_add  Device i2c address(8bit)
        @param[in]  reg_add     register address(8bit)
        @return char read data
    */
    virtual char read_reg(char Device_add, char reg_add);

    /**
        @brief single write register.
        @param[in]  Device_add  Device i2c address(8bit)
        @param[in]  reg_add     register address(8bit)
        @param[in]  data        write data
        @return int write result(define mbed(i2c))
    */
    virtual int write_reg(char Device_add, char reg_add, char data);
    
    private:
    I2C *i2c_p;
    I2C &i2c;
    
};

#endif