// -*- coding: utf-8 -*-
/**
 @file      GPS_INT.h
 @brief     Recieve GPS using Interrupt

 
 @author    D.Nakayama
 @version   1.0
 @date      2018-07-08  D.Nakayama  Written for C++/mbed.
 
 
 @see 
 Copyright (C) 2018 D.Nakayama.
 Released under the MIT license.
 http://opensource.org/licenses/mit-license.php
 
*/

#ifndef INCLUDED_GPS_INT_h_
#define INCLUDED_GPS_INT_h_
#include "mbed.h"

/**
    @class  GPS_INT
    @brief    "Recieve GPS class" using Interrupt
    @attention Since it is received by interrupt processing, another work can be done in the main loop.
    @attention Examples of character strings sent from the GPS sent by UART
    @attention $GPGGA,085120.307,3541.1493,N,13945.3994,E,1,08,1.0,6.9,M,35.9,M,,0000*5E
    @attention $GPGSA,A,3,29,26,05,10,02,27,08,15,,,,,1.8,1.0,1.5*3E
    @attention $GPRMC,085120.307,A,3541.1493,N,13945.3994,E,000.0,240.3,181211,,,A*6A
    @attention check board  : NUCLEO-F103RB, LPC1768
    @attention check device : GMS7-CR6(SIRF-IV)
 */
class GPS_INT{
public:

    //コンストラクタ
    /**
         @brief Create a new gps(UART).
         @param tx          Tx pin name (Defined in PinName.h)
         @param rx          Rx pin name (Defined in PinName.h)
         @param baudrate    Baudrate (ex: 4800). Default value is 9600.
    */
    GPS_INT(PinName tx, PinName rx, int baud = 9600);
    //デストラクタ
    /**
         @brief Disable the gps Port.
         @param No parameters.
    */
    virtual ~GPS_INT();
    /**
         @brief Whether location information has been updated.
         @param No parameters.
    */
    bool location_is_update(void);      //位置情報が更新されたか(緯度，経度，DOP)
    /**
         @brief Whether all infomation has been updated.
         @param No parameters.
    */
    bool info_is_update(void);          //すべての情報が更新されたか
    /**
         @brief Whether GPS is lock.
         @param No parameters.
    */
    bool is_lock(void);        //GPSがロックしたか

    //受信する情報
    /**
         @brief UTC
    */
    struct tm t;            //時刻UTC
    /**
         @brief UTC Elapsed seconds since January 1, 1900
    */
    time_t seconds;     //時刻(UTC1900年1月1日からの経過時間)
    /**
         @brief Longitude
    */
    double lon;         //緯度, 度(北緯が正)
    /**
         @brief Latitude
    */
    double lat;         //経度, 度(東経が正)
    /**
         @brief Location specific quality (0: Can not locate, 1: SPS (standard positioning service), 2: differencial GPS)
    */
    volatile int lock;  //位置特定品質(0:位置特定できない, 1:SPS(標準測位サービス), 2:differencial GPS)
    /**
         @brief Number of satellites used
    */
    int n_sat;          //使用衛星数
    /**
         @brief Horizontal Dilution Of Precision
    */
    float HDOP;         //水平精度低下率
    /**
         @brief Vertical Dilution Of Precision
    */
    float VDOP;         //垂直精度低下率
    /**
         @brief Position Dilution Of Precision
    */
    float PDOP;         //位置精度低下率
    /**
         @brief Antenna sea level height, m
    */
    float h_see;        //アンテナ海抜高さ, m
    /**
         @brief Geoid height, m
    */
    float h_geo;        //ジオイド高さ, m

private:
    Serial gps;
    bool location_update;
    bool info_update;
    int date_raw;       //UTC 日付生データ    
    float time_raw;     //UTC 時刻生データ

    double lon_raw;     //緯度生データ    
    int lon_int;        //緯度整数部
    double lon_minute;  //緯度分
    char ns;            //北緯or南緯

    double lat_raw;     //経度生データ
    int lat_int;        //経度整数部
    double lat_minute;  //経度分
    char ew;            //東経or西経

    void get_char(void);
    void gps_update(char* buffer);
};

#endif