// -*- coding: utf-8 -*-
/**
 @file      BME280.h
 @brief     MBE280のライブラリ
 
 @author    D.Nakayama
 @version   1.0
 @date      2018-07-10  D.Nakayama  Written for C++/mbed.
 
 
 @see 
 Copyright (C) 2018 D.Nakayama.
 Released under the MIT license.
 http://opensource.org/licenses/mit-license.php
 
*/

#ifndef BME280_LIB_H
#define BME280_LIB

#include "mbed.h"
#include "i2c_general_io.h"

#include "BME280_reg.h"     //レジスタ関連の定義

/**
    @class  BME280
    @brief  i2cでBME280にアクセスするためのクラス
*/
class BME280{

public:
    /**
         @brief BME280クラスの作成
         @param[in] sda             SDAピン名 (PinName.hに定義)
         @param[in] sck             SCKピン名 (PinName.hに定義)
         @param[in] mode            測定動作モード(BME_reg.hに定義)
         @param[in] press_sample    気圧のオーバーサンプリング(BME_reg.hに定義)
         @param[in] temp_sample     温度のオーバーサンプリング(BME_reg.hに定義)
         @param[in] hum_sample      湿度のオーバーサンプリング(BME_reg.hに定義)
         @param[in] filter          IIRフィルタの時定数設定(BME_reg.hに定義)
         @param[in] stanby          測定後休止時間の設定(ノーマルモードのみ有効, BME_reg.hに定義)         
    */
    BME280(PinName sda, PinName scl, 
    char mode           = BME280_FORCE_MODE, 
    char press_sample   = BME280_PRESS_OVER_SAMPL1, 
    char temp_sample    = BME280_TEMP_OVER_SAMPL1, 
    char hum_sample     = BME280_HUM_OVER_SAMPL1,
    char filter         = BME280_IIR_OFF,
    char stanby         = 0);

    /**
         @brief BME280クラスの作成.
         @param &i2c_obj            i2c名
         @param[in] mode            測定動作モード(BME_reg.hに定義)
         @param[in] press_sample    気圧のオーバーサンプリング(BME_reg.hに定義)
         @param[in] temp_sample     温度のオーバーサンプリング(BME_reg.hに定義)
         @param[in] hum_sample      湿度のオーバーサンプリング(BME_reg.hに定義)
         @param[in] filter          IIRフィルタの時定数設定(BME_reg.hに定義)
         @param[in] stanby          測定後休止間の設定(ノーマルモードのみ有効, BME_reg.hに定義)         
    */
    BME280(GEN_I2C &i2c_obj,
    char mode           = BME280_FORCE_MODE, 
    char press_sample   = BME280_PRESS_OVER_SAMPL1, 
    char temp_sample    = BME280_TEMP_OVER_SAMPL1, 
    char hum_sample     = BME280_HUM_OVER_SAMPL1,
    char filter         = BME280_IIR_OFF,
    char stanby         = 0);
    
    /**
         @brief BME280クラスのデストラクタ
         @param パラメータなし.
    */
    virtual ~BME280();


    /**
         @brief 気圧(単位:Pa)
    */
    double press;
    /**
         @brief 温度(単位:℃)
    */
    double temp;
    /**
         @brief 湿度(単位:％)
    */
    double hum;

    
    /**
         @brief センサからデータを読み取って気圧、温度、湿度データ更新
         @param パラメータなし.
    */
    void read_sensor(void);

private:

    //i2c用
    GEN_I2C *i2c_p;
    GEN_I2C &sensor;

    //初期化関数
    void init(void);    

    //設定保持用
    char _mode;
    char _press_sample;
    char _temp_sample;
    char _hum_sample;
    char _filter;
    char _stanby;

    //強制測定モードをセットする関数。
    inline void set_force_mode(void);

//データ補正用関数とそこで使う変数(データシートより)
    long signed int t_fine;
    long signed int BME280_compensate_T_int32(long signed int adc_T);
    long unsigned int BME280_compensate_P_int64(long signed int adc_P);
    long unsigned int BME280_compensate_H_int32(long signed int adc_H);

    //温度補正係数
    unsigned short dig_T1;
    signed short   dig_T2;
    signed short   dig_T3;

    //気圧補正係数
    unsigned short dig_P1;
    signed short   dig_P2;
    signed short   dig_P3;
    signed short   dig_P4;
    signed short   dig_P5;
    signed short   dig_P6;
    signed short   dig_P7;
    signed short   dig_P8;
    signed short   dig_P9;

    //湿度補正係数
    unsigned char  dig_H1;
    signed short   dig_H2;
    unsigned char  dig_H3;
    signed short   dig_H4;
    signed short   dig_H5;
    signed char    dig_H6;

};

#endif