/**
*@file main.h
*
*@brief Library for enabling the main.cpp of the METEOR DODGER game to work.
*@brief So that the code will work, the definitions, variables, parameters.
*@brief and other libraries will need to be included and stated.
*
*@date 10/05/2015
*
*/


#ifndef MAIN_H
#define MAIN_H

/**
*@brief List of require libraries that enable lines of code to function.
*@brief Some lines of code have words that execute integers of other functions.
*@brief These are substituted in the definitions section.
*@author Jeremy Ogus
*@date 10/05/2015
*/

//libraries included
#include "mbed.h"
#include "N5110.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"

//stated definitions
#define DIRECTION_TOLERANCE 0.15 
#define getTo goto
#define zipTo goto

/**
*@namespace lcd
*@brief output pins for the LCD displace.
*/
//Nokia LCD inputs, VCC,SCE,RST,D/C,MOSI,SCLK,LED
N5110 lcd(p7,p8,p9,p10,p11,p13,p21); 

/**
*@namespace leds, led1, led3, led4
*@brief internal mbed leds used in the intro to signify powerup.
*/
//LED inputs on mbed
BusOut leds(LED4, LED3, LED2, LED1);
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);

/**
*@namespace buttonA, buttonB
*@brief inputs for the two push buttons.
*/
//push buttons A & B inputs
DigitalIn buttonA(p16);
InterruptIn buttonB(p17);

/**
*@namespace button, xPot, yPot
*@brief inputs to enable the joystick to funciton.
*/
//joystick inputs
DigitalIn button(p18);
AnalogIn xPot(p19);
AnalogIn yPot(p20);

/**
*@namespace buzzer
*@brief input on PWM for the buzzer.
*/
//buzzer input
PwmOut buzzer(p22);

/**
*@param meteorstart - interger to check state whilst spawning the AI.
*@param magchange - interger to check whether to break out of game loop when the charcer
*@param pxpos - interger to control position for both the character and AI in the horizontal direction
*@param pypos - interger to control position for both the character and AI in the vertical direction
*@param mxpos - interger to create array to limit hostile AI on display
*@param mypos - interger to create array to limit hostile AI on display
*@param goneFishing - timer to turn enter deep power down 10 seconds after game end, with no user input to replay
*@param enable - loop for pause system, interrupt by button B
*/

//intergers for AI loop system
int meteorstart = 1;
int magchange;

//integers for user position
int pxpos;
int pypos;  

// int for AI array
int mxpos[5], mypos[5]; 


Timer goneFishing;

void enable(void);


//int joyCont();     RIP, was used in an earlier version. still here for personal reference


/*
*____________________________________________________Definitions and inputs END_____________________________________________________________________
*
*
*
*____________________________________________________Joystick (variables, definition, sensor) START___________________________________________________
*/

/**
*@param pollJosytick - timer to constantly read joystick sensor 20 times/second
*@param DirectionName - ironically, defines direction names when listed below in the if direction loop
*@brief the remaining code is self explanatory, reading it will tell you what it does, however trivial it seems
*/

// timer to regularly read the joystick
Ticker pollJoystick;


// create enumerated type (0,1,2,3 etc. for direction)
enum DirectionName {
    UP,
    DOWN,
    LEFT,
    RIGHT,
    UP_RIGHT,
    UP_LEFT,
    DOWN_RIGHT,
    DOWN_LEFT,
    CENTRE,
    UNKNOWN
};

// struct for Joystick
typedef struct JoyStick Joystick;
struct JoyStick {
    float x;    // current x value
    float x0;   // 'centred' x value
    float y;    // current y value
    float y0;   // 'centred' y value
    int button; // button state (assume pull-down used, so 1 = pressed, 0 = unpressed)
    DirectionName direction;  // current direction
};
// create struct variable
Joystick joystick;

int printFlag = 0;

// function prototypes
void calibrateJoystick();
void updateJoystick();


// read default positions of the joystick to calibrate later readings
void calibrateJoystick()
{
    button.mode(PullUp);
    // must not move during calibration
    joystick.x0 = xPot;  // initial positions in the range 0.0 to 1.0 (0.5 if centred exactly)
    joystick.y0 = yPot;
}
void updateJoystick()
{
    // read current joystick values relative to calibrated values (in range -0.5 to 0.5, 0.0 is centred)
    joystick.x = xPot - joystick.x0;
    joystick.y = yPot - joystick.y0;
    // read button state
    joystick.button = button;

    // calculate direction depending on x,y values
    // tolerance allows a little lee-way in case joystick not exactly in the stated direction
    if ( fabs(joystick.y) < DIRECTION_TOLERANCE && fabs(joystick.x) < DIRECTION_TOLERANCE) {
        joystick.direction = CENTRE;
    } 
    else if ( joystick.y > DIRECTION_TOLERANCE && fabs(joystick.x) < DIRECTION_TOLERANCE) {
        joystick.direction = UP;
    } 
    else if ( joystick.y < DIRECTION_TOLERANCE && fabs(joystick.x) < DIRECTION_TOLERANCE) {
        joystick.direction = DOWN;
    } 
    else if ( joystick.x > DIRECTION_TOLERANCE && fabs(joystick.y) < DIRECTION_TOLERANCE) {
        joystick.direction = LEFT;
    }
    else if ( joystick.x < DIRECTION_TOLERANCE && fabs(joystick.y) < DIRECTION_TOLERANCE) {
        joystick.direction = RIGHT;
    } 
    else if ( joystick.x < DIRECTION_TOLERANCE && joystick.y > DIRECTION_TOLERANCE) {
        joystick.direction = UP_RIGHT;
    }
    else if ( joystick.x > DIRECTION_TOLERANCE && joystick.y > DIRECTION_TOLERANCE) {
        joystick.direction = UP_LEFT;
    }
    else if ( joystick.x < DIRECTION_TOLERANCE && joystick.y < DIRECTION_TOLERANCE) {
        joystick.direction = DOWN_RIGHT;
    }
    else if ( joystick.x > DIRECTION_TOLERANCE && joystick.y < DIRECTION_TOLERANCE) {
        joystick.direction = DOWN_LEFT;
    }   
     
    else {
        joystick.direction = UNKNOWN;
    }

    // set flag for printing
    printFlag = 1;
}

/*
*____________________________________________________Joystick (variables, definition, sensor) END___________________________________________________
*
*
*____________________________________________________Pause game interrupt START______________________________________________________________________
*/

/**
*@brief this section of code is a pause section. 
*@brief when B is pushed, code is paused.
*@brief when B is pushed again, code is resumed.
*@author Jeremy Ogus
*@date 10/05/2015
*@param buttonB - Push button, can have a state of either 0 or 1
*/

//when enabled, short delay. This is so you don't switch the state
//of the button by the time code has a chance to loop. Should take 
//less than 0.2 seconds for the user to press buttonB and let go.
void enable(){
    wait(0.2);
    //if buttonB is pushed, wait 0.5s. This is for release time, as above
    if(buttonB==1){
        wait(0.5);
        //wait forever until buttonB is pushed again
        while(buttonB==1){
            }
        wait(0.2); //timer for release of the button to resume
            
    }     
    
}

#endif