#include "mbed.h"

#include "BREMSConfig.h"
#include "BREMSStructs.h"
#include "BufferedLogger.h"
#include "CommandProcessor.h"
#include "PreferenceWriter.h"
#include "Filter.h"
#include "LedBlinker.h"

#include "hardware.h"
#include "derived.h"
#include "prefs.h"
#include "errors.h"

void BREMSConfigRegisters(IOStruct *io) {
    RCC->AHB1ENR |= RCC_AHB1ENR_GPIOAEN;
    RCC->AHB1ENR |= RCC_AHB1ENR_GPIOBEN;
    RCC->AHB1ENR |= RCC_AHB1ENR_GPIOCEN;
    
    RCC->APB2ENR |= RCC_APB2ENR_TIM1EN; //enable TIM1 clock
    
    io->a = new FastPWM(PWMA);
    io->b = new FastPWM(PWMB);
    io->c = new FastPWM(PWMC);
    
    NVIC_EnableIRQ(TIM1_UP_TIM10_IRQn); //Enable TIM1 IRQ

    TIM1->DIER |= TIM_DIER_UIE; //enable update interrupt
    TIM1->CR1 = 0x00; //CMS = 10, interrupt only when counting up
    TIM1->CR1 |= TIM_CR1_ARPE; //autoreload on, 
    TIM1->RCR |= 0x00; //update event once per up/down count of tim1 
    TIM1->EGR |= TIM_EGR_UG;
    
    TIM1->PSC = 0x00; //no prescaler, timer counts up in sync with the peripheral clock
    TIM1->ARR = (int) (2 * (float) 9e7 / _F_SW);
    TIM1->CCER |= (TIM_CCER_CC1NP); //rising edge aligned, non-inverting
    TIM1->CR1 |= TIM_CR1_CEN;
    
    //ADC Setup
    RCC->APB2ENR |= RCC_APB2ENR_ADC1EN; // clock for ADC1
    RCC->APB2ENR |= RCC_APB2ENR_ADC2EN; // clock for ADC2
    
    ADC->CCR = 0x00000006; //Regular simultaneous mode, 3 channels
    
    ADC1->CR2 |= ADC_CR2_ADON; //ADC1 on
    ADC1->SQR3 = 0x0000004; //PA_4 as ADC1, sequence 0
    
    ADC2->CR2 |= ADC_CR2_ADON; //ADC2 ON
    ADC2->SQR3 = 0x00000008; //PB_0 as ADC2, sequence 1
    
    //PA_4 as analog
    GPIOA->MODER |= (1 << 8);
    GPIOA->MODER |= (1 << 9);
    
    //PB_0 as analog
    GPIOB->MODER |= (1 << 0);
    GPIOB->MODER |= (1 << 1);
    
    //DAC setup
    RCC->APB1ENR |= 0x20000000;
    DAC->CR |= DAC_CR_EN2;
    
    //PA_5 as analog
    GPIOA->MODER |= (1 << 10);
    GPIOA->MODER |= (1 << 11);
    
    set_dtc(io->a, 0.0f);
    set_dtc(io->b, 0.0f);
    set_dtc(io->c, 0.0f);
}

void BREMSZeroCurrent(ReadDataStruct *read) {
    for (int i = 0; i < 1000; i++){
        read->ad1_supp_offset += (float) (ADC1->DR);
        read->ad2_supp_offset += (float) (ADC2->DR);
        ADC1->CR2  |= 0x40000000;
        wait_us(100); 
    }
    read->ad1_supp_offset /= 1000.0f;
    read->ad2_supp_offset /= 1000.0f;
    read->ad1_supp_offset = read->ad1_supp_offset / 4096.0f * AVDD - I_OFFSET;
    read->ad2_supp_offset = read->ad2_supp_offset / 4096.0f * AVDD - I_OFFSET;
}

void BREMSStartupMsg(ReadDataStruct *read, Serial *pc) {
    pc->printf("%s\n", "FOC'ed in the Bot Rev A.");
}

void BREMSInit(IOStruct *io, ReadDataStruct *read, FOCStruct *foc, ControlStruct *control, bool tune) {    
    io->en = new DigitalOut(EN);
    io->en->write(0);
    
    io->pc = new Serial(USBTX, USBRX);
    io->pc->baud(115200);
    NVIC_SetPriority(USART2_IRQn, 2);
    
    init_masks();
    
    wait_ms(50);
    DigitalOut resolver_reset_out(RESOLVER_RESET);
    resolver_reset_out = 0;
    wait_ms(10);
    DigitalIn resolver_reset_in(RESOLVER_RESET);
    
    cmd_clear(io->pc);
    BREMSStartupMsg(read, io->pc);
    
    BREMS_mode = MODE_CFG;    
    io->pref = new PreferenceWriter(6);
    cmd_reload(io->pc, io->pref);
    if (_PREFS_VALID != 1) {
        io->pc->printf("%s\n", "Stored config invalid");
        BREMS_mode = MODE_CFG;
        cmd_defaults(io->pc);
        BREMS_op = OP_TORQUE;
        BREMS_src = CMD_SRC_TERMINAL;
        io->pc->printf("%s\n", "You should probably at least set throttle and current limits!");
    }
    
    io->blink = new LedBlinker(STATUS_LED, _F_SW, 1.0);
    io->blink->set_code(0x0f);
    
    wait_ms(750);

    io->pos = new PositionSensorEncoder(_CPR, 0);
    io->logger = new BufferedLogger(_LOG_PACKET_SIZE, (_LOG_PAGE_SIZE-_LOG_HEADER_SIZE)/(_LOG_PACKET_SIZE+1), LOG_TX, LOG_RX, _LOG_BAUD_RATE);
    io->throttle_in = new PwmIn(TH_PIN, _TH_LIMIT_LOW, _TH_LIMIT_HIGH);
    
    io->cmd_busy = false;
    
    control->throttle_filter = new MedianFilter(_THROTTLE_FILTER_WINDOW);
    control->velocity_filter = new MedianFilter(_W_FILTER_WINDOW);
    
    read->vbus = _BUS_VOLTAGE;
    read->w = 0.0f;
    read->ad1_supp_offset = 0.0f;
    read->ad2_supp_offset = 0.0f;
    read->p_mech = io->pos->GetMechPosition();
    
    BREMSConfigRegisters(io);
    wait_ms(250);
    BREMSZeroCurrent(read);
    io->pc->printf("%s", ">");
    
    control->d_integral = 0.0f;
    control->q_integral = 0.0f;
    control->d_filtered = 0.0f;
    control->q_filtered = 0.0f;
    control->last_d = 0.0f;
    control->last_q = 0.0f;
    control->d_ref = 0.0f;
    control->q_ref = 0.0f;
    control->torque_percent = 0.0f;
    control->w_integral = 0.0f;
    control->enabled = false;

    io->en->write(1);
}