/******************************************************************************
* MIT License
*
* Copyright (c) 2017 Justin J. Jordan
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
 
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
******************************************************************************/


#ifndef _MULTIPINRGB_H_
#define _MULTIPINRGB_H_

#include "mbed.h"


/**
* @brief MutlipinRGB Library
*
* @details Library for multi-pin RGB leds that encapsulates three PwmOut objects 
* and provides access to the floating point read/write mbr fxs of the PwmOut 
* objects.\n\n
* Duty cycles should always be written as 0.0F for off, and 1.0F as 100% on,
* regardless of led active state.\n\n
* Duty cycles are reported in the same manner.\n\n 
*
* @code
* #include "mbed.h"
* #include "MultipinRGB.h"
*
* int main ()
* {
*   MultipinRGB leds(LED1, LED2, LED3);
*   float redDutyCycle(0.5F), grnDutyCycle(0.0F), bluDutyCycle(0.0F), temp;
*    
*   while(1)
*   {
*     leds.writeLeds(redDutyCycle, grnDutyCycle, bluDutyCycle);
*
*     printf("RGB Duty Cycles = %3.1f, %3.1f, %3.1f\r\n", 
*             redDutyCycle, grnDutyCycle, bluDutyCycle);
*
*     //shift r->g->b->r
*     temp = bluDutyCycle;
*     bluDutyCycle = grnDutyCycle;
*     grnDutyCycle = redDutyCycle;
*     redDutyCycle = temp;
*      
*     wait(0.25);
*   }
* }
* @endcode
*
*/
class MultipinRGB
{
    public:
    
    enum Led_e
    {
        Red,
        Green,
        Blue
    };
    
    enum LedLogic_e
    {
        ActiveLow,
        ActiveHigh
    };
    
    ///Constructor
    ///@param[in] red - Pin that red led is connected to.
    ///@param[in] green - Pin that green led is connected to.
    ///@param[in] blue - Pin that blue led is connected to.
    ///@param[in] activeState - Active state of all three leds.
    MultipinRGB(PinName red, PinName green, PinName blue, LedLogic_e 
    activeState = ActiveLow);
    
    ///Destructor
    ~MultipinRGB();
    
    ///@brief Sets duty cycle for led.\n
    ///
    ///On Entry:
    ///@param[in] led - Led to update
    ///@param[in] dc - Duty cycle for led, 0.0 to 1.0 
    ///
    ///On Exit:
    ///@param[out] none
    ///
    ///@returns none
    void writeLed(const Led_e led, const float dc);
    
    ///@brief Sets duty cycle for all three leds.\n
    ///
    ///On Entry:
    ///@param[in] r - Duty cycle for led, 0.0 to 1.0  
    ///@param[in] g - Duty cycle for led, 0.0 to 1.0  
    ///@param[in] b - Duty cycle for led, 0.0 to 1.0  
    ///
    ///On Exit:
    ///@param[out] none
    ///
    ///@returns none
    void writeLeds(const float r, const float g, const float b);
    
    ///@brief Reads duty cycle for led.\n
    ///
    ///On Entry:
    ///@param[in] led - Led to update
    ///
    ///On Exit:
    ///@param[out] none
    ///
    ///@returns Current duty cycle for led, 0.0 to 1.0  
    float readLed(const Led_e led);
    
    ///@brief Reads duty cycle for all three leds.\n
    ///
    ///On Entry:
    ///@param[in] r - float reference for red led duty cycle.
    ///@param[in] g - float reference for green led duty cycle.
    ///@param[in] b - float reference for blue led duty cycle.
    ///
    ///On Exit:
    ///@param[out] r - Current duty cycle for led, 0.0 to 1.0   
    ///@param[out] g - Current duty cycle for led, 0.0 to 1.0   
    ///@param[out] b - Current duty cycle for led, 0.0 to 1.0   
    ///
    ///@returns none
    void  readLeds(float& r, float& g, float& b);
    
    ///@brief Toggles led from off to on, or on to off.\n
    ///Duty Cycle will be 0% or 100% after this call.\n
    ///
    ///On Entry:
    ///@param[in] led - Led to toggle 
    ///
    ///On Exit:
    ///@param[out] none
    ///
    ///@returns none
    void toggleLed(const Led_e led);
    
    ///@brief Sets pwm period for all three leds.\n
    ///
    ///On Entry:
    ///@param[in] p - PWM period in seconds 
    ///
    ///On Exit:
    ///@param[out] none
    ///
    ///@returns none
    void setPeriod(const float p);
    
    private:
    
    PwmOut m_red;
    PwmOut m_green;
    PwmOut m_blue;
    
    LedLogic_e m_ledActiveState;
};


///@brief fx documentation template.\n
///
///On Entry:
///@param[in] none 
///
///On Exit:
///@param[out] none
///
///@returns none

#endif /*_MULTIPINRGB_H_ */