/******************************************************************************
* MIT License
*
* Copyright (c) 2017 Justin J. Jordan
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
 
* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.
 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
******************************************************************************/


#include "MultipinRGB.h"


/*****************************************************************************/
MultipinRGB::MultipinRGB(PinName red, PinName green, PinName blue, LedLogic_e activeState):
m_red(red), m_green(green), m_blue(blue), m_ledActiveState(activeState)
{
    //turn off all leds, use default period
    if(m_ledActiveState == MultipinRGB::ActiveLow)
    {
        m_red.write(1.0F);
        m_green.write(1.0F);
        m_blue.write(1.0F);
    }
    else
    {
        m_red.write(0.0F);
        m_green.write(0.0F);
        m_blue.write(0.0F);
    }
    
    
}


/*****************************************************************************/
MultipinRGB::~MultipinRGB()
{
    //empty block
}


/*****************************************************************************/
void MultipinRGB::writeLed(const Led_e led, const float dc)
{
    switch(led)
    {
        case(Red):
            //if
            (m_ledActiveState == MultipinRGB::ActiveLow) ? 
            m_red.write(1.0F - dc) : m_red.write(dc);
        break;
        
        case(Green):
            //if
            (m_ledActiveState == MultipinRGB::ActiveLow) ? 
            m_green.write(1.0F - dc) : m_green.write(dc);
        break;
        
        case(Blue):
            //if
            (m_ledActiveState == MultipinRGB::ActiveLow) ? 
            m_blue.write(1.0F - dc) : m_blue.write(dc);
        break;
        
        default:
            mbed_die();
        break;
    };
}


/*****************************************************************************/
void MultipinRGB::writeLeds(const float r, const float g, const float b)
{
    if(m_ledActiveState == MultipinRGB::ActiveLow)
    {
        m_red.write(1.0F - r);
        m_green.write(1.0F - g);
        m_blue.write(1.0F - b);
    }
    else
    {
        m_red.write(r);
        m_green.write(g);
        m_blue.write(b);
    }
}


/*****************************************************************************/
float MultipinRGB::readLed(const Led_e led)
{
    float rtnVal;
    switch(led)
    {
        case(Red):
            //if
            (m_ledActiveState == MultipinRGB::ActiveLow) ? 
            (rtnVal = (1.0F - m_red.read())) : (rtnVal = m_red.read());
        break;
        
        case(Green):
            //if
            (m_ledActiveState == MultipinRGB::ActiveLow) ? 
            (rtnVal = (1.0F - m_green.read())) : (rtnVal = m_green.read());
        break;
        
        case(Blue):
            //if
            (m_ledActiveState == MultipinRGB::ActiveLow) ? 
            (rtnVal = (1.0F - m_blue.read())) : (rtnVal = m_blue.read());
        break;
        
        default:
            mbed_die();
        break;
    };
    
    return rtnVal;
}


/*****************************************************************************/
void  MultipinRGB::readLeds(float& r, float& g, float& b)
{
    if(m_ledActiveState == MultipinRGB::ActiveLow)
    {
        r = (1.0F - m_red.read());
        g = (1.0F - m_green.read());
        b = (1.0F - m_blue.read());
    }
    else
    {
        r = m_red.read();
        g = m_green.read();
        b = m_blue.read();
    }
}


/*****************************************************************************/
void MultipinRGB::toggleLed(const Led_e led)
{
    float currentDc, newDc;
    
    switch(led)
    {
        case(Red):
            currentDc = m_red.read();
        break;
        
        case(Green):
            currentDc = m_green.read();
        break;
        
        case(Blue):
            currentDc = m_blue.read();
        break;
        
        default:
            mbed_die();
        break;
    };
    
    if(currentDc >= 1.0F)
    {
        newDc = 0.0F;
    }
    else if(currentDc <= 0.0F)
    {
        newDc = 1.0F;
    }
    else
    {
        newDc = (m_ledActiveState == MultipinRGB::ActiveLow) ? 1.0F : 0.0F;
    }
    
    switch(led)
    {
        case(Red):
            m_red.write(newDc);
        break;
        
        case(Green):
            m_green.write(newDc);
        break;
        
        case(Blue):
            m_blue.write(newDc);
        break;
        
        default:
            mbed_die();
        break;
    };
}


/*****************************************************************************/
void MultipinRGB::setPeriod(const float p)
{
    m_red.period(p);
    m_green.period(p);
    m_blue.period(p);
}
