/******************************************************************************
* MIT License
*
* Copyright (c) 2017 Justin J. Jordan
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:

* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.

* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
******************************************************************************/

#include "mbed.h"
#include "OneWire.h"
#include "OWGridEye.h"
#include "MAX4822.h"
#include "ds3231.h"

using namespace OneWire;
using namespace RomCommands;

volatile uint8_t seconds = 0;
void incrementSeconds()
{
    seconds++;
}



int main()
{
    //Get I2C bus instance
    I2C i2c(D14, D15);
    i2c.frequency(400000);
    
    //Get SPI bus instance
    SPI spi_bus(D11, D12, D13);
    //Get MAX4822 instance
    MAX4822 rly_drvr(spi_bus, D10);
    //Set/Reset all pins
    DigitalOut rly_drvr_set(D7, 1);
    DigitalOut rly_drvr_reset(D6, 1);
    
    rly_drvr.reset_all_relays(rly_drvr_reset);
    
    //Get 1-Wire Master instance, OWM
    DS2484 owm(i2c);
    OneWireMaster::CmdResult owm_result = owm.OWInitMaster();
    while(owm_result != OneWireMaster::Success)
    {
        printf("Failed to init OWM...\r\n");
        owm_result = owm.OWInitMaster();
    }
    printf("OWM Initialized!!!\r\n");
    
    //Struct for encapsulating search algorithm vars/state
    SearchState searchState;
    
    //Encapsulate the OWM in a Bus Iterator
    MultidropRomIterator selector(owm);
    //Give the iterator to the sensor.
    //Now we can work directly with the sensor instead of through the OWM
    OWGridEye sensor(selector);
    
    //Search bus, expecting three devices; 
    //DS2431 on RD130, DS2413 and DS28E17 on RD131
    owm_result = OWFirst(owm, searchState);
    if(owm_result == OneWireMaster::Success)
    {
        do
        {
            if(searchState.romId.familyCode() == OWGridEye::DS2413_FAMILY_CODE)
            {
                sensor.setOWSwitchRomId(searchState.romId);
                sensor.connectGridEye();
            }
            
            if(searchState.romId.familyCode() == OWGridEye::DS28E17_FAMILY_CODE)
            {
                sensor.setI2CBridgeRomId(searchState.romId);
                sensor.disconnectGridEye();
            }
            
            owm_result = OWNext(owm, searchState);
        }
        while(owm_result == OneWireMaster::Success);
    }
    else
    {
        printf("Failed to find any 1-Wire devices...\r\n");
    }
    
    const uint8_t NUM_PIXELS = 64;
    const float ON_TIME = (1.0F * 60.0F);  //Replace first literal with number of minutes
    
    int16_t dataBuffer[NUM_PIXELS];
    uint8_t idx(0), referenceCount(0);
    float sumTemp, avgTemp, referenceTemp;
    
    Ticker secondTimer;
    secondTimer.attach(&incrementSeconds, 1);
    
    do
    {
        if(seconds == 1)
        {
            seconds = 0;
            sensor.connectGridEye();
            
            wait(0.1); //Let sensor power up
            sensor.gridEyeGetFrameTemperature(dataBuffer);
            sensor.disconnectGridEye();
            
            sumTemp = 0.0F;
            for(idx = 0; idx < NUM_PIXELS; idx++)
            {
                sumTemp += fAMG_PUB_CMN_ConvStoF(dataBuffer[idx]);
            }
            avgTemp = (sumTemp / (1.0F * NUM_PIXELS));
            printf("Average Temperature = %4.1f\r\n", avgTemp);
            
            referenceTemp += avgTemp;
            referenceCount++;
        }
    }
    while(referenceCount < 15);
    referenceTemp = ((referenceTemp/15.0F) + 1.0F);
    printf("Reference Temperature = %4.1f\r\n", referenceTemp);
    
    Timer onTimer;
    onTimer.start();
    
    while(1)
    {
        if(seconds == 1)
        {
            seconds = 0;
            sensor.connectGridEye();
            
            wait(0.1); //Let sensor power up
            sensor.gridEyeGetFrameTemperature(dataBuffer);
            sensor.disconnectGridEye();
            
            sumTemp = 0.0F;
            for(idx = 0; idx < NUM_PIXELS; idx++)
            {
                sumTemp += fAMG_PUB_CMN_ConvStoF(dataBuffer[idx]);
            }
            avgTemp = (sumTemp / (1.0F * NUM_PIXELS));
            printf("Average Temperature = %4.1f\r\n", avgTemp);
            printf("Seconds since last reset = %0.1f\r\n\r\n", onTimer.read());
            
            //If average temperature is greater than reference, turn on relays.
            //Assumes object detected
            if(avgTemp > referenceTemp)
            {
                onTimer.reset();
                rly_drvr.set_all_relays(rly_drvr_set);
                
            }
        }
        
        //If average is less than reference for ON_TIME, shut relays off
        if(onTimer.read() > ON_TIME)
        {
            rly_drvr.reset_all_relays(rly_drvr_reset);
        }
    }
}
