/******************************************************************************
* MIT License
*
* Copyright (c) 2017 Justin J. Jordan
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:

* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.

* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
******************************************************************************/


#include "mbed.h"
#include "OneWire.h"
#include "max32630fthr.h"
#include "SDFileSystem.h"

#define TERM_HOME "\033[H"
#define TERM_CLEAR_FROM_CURSOR "\033[0J"
#define TERM_CLEAR_EOL "\033[K"

using namespace OneWire;
using namespace RomCommands;


Ticker oneSecondTicker;
volatile uint32_t secondCounter = 0;
void oneSecond()
{
    secondCounter++;
}

//Setup start/stop button
DigitalIn sw2(P2_3, PullUp);
InterruptIn startStopLog(P2_3);
volatile bool startLog = false;
void startStopLogISR()
{
    startLog = !startLog;
}


int main ()
{
    //Init board and set GPIO to 3.3V logic
    MAX32630FTHR pegasus;
    pegasus.init(MAX32630FTHR::VIO_3V3);
    
    //Turn RGB LED off
    DigitalOut rLED(LED1, LED_ON);
    DigitalOut gLED(LED2, LED_OFF);
    DigitalOut bLED(LED3, LED_OFF);
    
    //Clear terminal screen
    printf(TERM_HOME);
    printf(TERM_CLEAR_FROM_CURSOR);
    
    //Get 1-Wire Master (owm) instance
    //         (extWeakPup, extStrongPup)
    MCU_OWM owm(false, true);
    
    //Make sure owm is initialized
    OneWireMaster::CmdResult result = owm.OWInitMaster();
    while(result != OneWireMaster::Success)
    {
        printf("Failed to init OWM...\r\n\r\n");
        result = owm.OWInitMaster();
        wait(0.5);
    }
    printf("OWM Initialized...\r\n");
    wait(2.0);
    printf(TERM_HOME);
    printf(TERM_CLEAR_FROM_CURSOR);
    
    
    //Check for DS18B20
    SearchState search_state;
    search_state.findFamily(0x28);
    do
    {
        result = OWNext(owm, search_state);
        if(search_state.romId.familyCode() != 0x28)
        {
            printf("Failed to find DS18B20...\r\n");
            printf("Please connect DS18B20...\r\n\r\n");
            wait(0.5);
        }
        else
        {
            printf("Found DS18B20...\r\n");
        }
    }
    while(search_state.romId.familyCode() != 0x28);
    wait(2.0);
    printf(TERM_HOME);
    printf(TERM_CLEAR_FROM_CURSOR);
    
    //Get instance of DS18B20 object
    MultidropRomIterator selector(owm);
    DS18B20 tempSensor(selector);
    tempSensor.setRomId(search_state.romId);
    
    
    //Get instance of sd card file system
    SDFileSystem sd(P0_5, P0_6, P0_4, P0_7, "sd");  // mosi, miso, sclk, cs
    
    //Configure pin for sd card detect
    DigitalIn uSDdetect(P2_2, PullUp);
    
    static const char FILE_NAME[] = "/sd/log.txt";
    FILE *fp;
    
    //Check for card
    if(uSDdetect)
    {
        printf("Please insert uSD card\r\n");
        while(uSDdetect);
    }
    
    rLED = LED_ON;
    gLED = LED_ON;
    do
    {
        fp = fopen(FILE_NAME, "w");
        if(fp != NULL)
        {
            printf("Card detected...\r\n");
            printf("Preparing log file...\r\n");
            fprintf(fp, "Time-Stamp, Battery Voltage, Temperature\n");
            fclose(fp);
        }
        else
        {
            printf("Failed to open file...\r\n");
            printf("Press Reset Button...\r\n\r\n");
            wait(0.5);
        }
    }
    while(fp == NULL);
    rLED = LED_OFF;
    gLED = LED_ON;
    
    wait(2.0);
    printf(TERM_HOME);
    printf(TERM_CLEAR_FROM_CURSOR);
    
    enum UiStates
    {
        Reset,
        NoLog_NoCardDetected,
        NoLog_CardDetected,
        Log_NoCardDetected,
        Log_CardDetected
    };

    UiStates currentState = Reset, oldState = Reset;
    
    static const uint32_t SAMPLE_INTERVAL = 5; // In seconds
    bool firstSampleTaken = false;
    uint32_t sampleTime = 0;
    float batteryVoltage, temperature;
    
    //attach timer to callback for 1 second tick
    oneSecondTicker.attach(&oneSecond, 1);
    
    //Tie SW2 to callback fx
    startStopLog.fall(&startStopLogISR);
    

    while(1)
    {
        //card detect is active low
        //Determine state, house keeping
        if(!startLog && uSDdetect) //False, False
        {
            currentState = NoLog_NoCardDetected;
            secondCounter = 0;
            sampleTime = 0;
            firstSampleTaken = false;
        }
        else if(!startLog && !uSDdetect) //False, True
        {
            currentState = NoLog_CardDetected;
            secondCounter = 0;
            sampleTime = 0;
            firstSampleTaken = false;
        }
        else if(startLog && uSDdetect) //True, False
        {
            currentState = Log_NoCardDetected;
            secondCounter = 0;
            sampleTime = 0;
            firstSampleTaken = false;
        }
        else //True, True
        {
            currentState = Log_CardDetected;
            
            //get t = 0 sample
            if(!firstSampleTaken)
            {
                firstSampleTaken = true;
                printf(TERM_HOME);
                printf(TERM_CLEAR_FROM_CURSOR);
                oldState = currentState;
                    
                pegasus.getBatteryVoltage(&batteryVoltage);
                tempSensor.convertTemperature(temperature);
                
                printf("Logging Data...\r\n\r\n");
                printf("Timestamp: %d (seconds)\r\n\r\n", sampleTime);
                printf("Battery Voltage = %s%3.2f\r\n\r\n", TERM_CLEAR_EOL, batteryVoltage);
                printf("Temperature = %s%5.2f\r\n\r\n", TERM_CLEAR_EOL, temperature);
                printf(TERM_HOME);
                
                rLED = LED_ON;
                fp = fopen(FILE_NAME, "a");
                if(fp != NULL)
                {
                    fprintf(fp, "%d, %f, %f\n", sampleTime, batteryVoltage, temperature);
                    fclose(fp);
                }
                else
                {
                    printf(TERM_HOME);
                    printf(TERM_CLEAR_FROM_CURSOR);
                    printf("Failed to open file\r\n");
                    printf("Press Reset Button...\r\n");
                    printf("MBED DIE!!...\r\n");
                    mbed_die();
                }
                rLED = LED_OFF;
            }
        }
        
        //If state has changed, clear dispaly
        if(oldState != currentState)
        {
            printf(TERM_HOME);
            printf(TERM_CLEAR_FROM_CURSOR);
            oldState = currentState;
        }
        
        //State based actions
        switch(currentState)
        {
            case NoLog_NoCardDetected:
                printf("Please insert uSD card...\r\n");
                printf(TERM_HOME);
            break;
            
            case NoLog_CardDetected:
                printf("Press SW2 to start logging data...\r\n");
                printf(TERM_HOME);
            break;
            
            case Log_NoCardDetected:
                startLog = false;
                printf("Please insert uSD card\r\n");
                printf(TERM_HOME);
            break;
            
            case Log_CardDetected:
            
                if(secondCounter >= SAMPLE_INTERVAL)
                {
                    secondCounter = 0;
                    sampleTime += SAMPLE_INTERVAL;
                    
                    pegasus.getBatteryVoltage(&batteryVoltage);
                    tempSensor.convertTemperature(temperature);
                    
                    printf("Logging Data...\r\n\r\n");
                    printf("Timestamp: %d (seconds)\r\n\r\n", sampleTime);
                    printf("Battery Voltage = %s%3.2f\r\n\r\n", TERM_CLEAR_EOL, batteryVoltage);
                    printf("Temperature = %s%5.2f\r\n\r\n", TERM_CLEAR_EOL, temperature);
                    printf(TERM_HOME);
                    
                    rLED = LED_ON;
                    fp = fopen(FILE_NAME, "a");
                    if(fp != NULL)
                    {
                        fprintf(fp, "%d, %f, %f\n", sampleTime, batteryVoltage, temperature);
                        fclose(fp);
                    }
                    else
                    {
                        printf(TERM_HOME);
                        printf(TERM_CLEAR_FROM_CURSOR);
                        printf("Failed to open file\r\n");
                        printf("Press Reset Button...\r\n");
                        printf("MBED DIE!!...\r\n");
                        mbed_die();
                    }
                    rLED = LED_OFF;
                }
                
            break;
            
            default:
                printf(TERM_HOME);
                printf(TERM_CLEAR_FROM_CURSOR);
                printf("Bad State\r\n");
                printf("Press Reset Button...\r\n");
                printf("MBED DIE!!...\r\n");
                mbed_die();
            break;
        }
    }    
}
