/******************************************************************************
* MIT License
*
* Copyright (c) 2017 Justin J. Jordan
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:

* The above copyright notice and this permission notice shall be included in all
* copies or substantial portions of the Software.

* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
******************************************************************************/


#include "H3LIS331DL.h"


//************************** Base Class member fxs *****************************
int32_t H3LIS331DL::getDeviceId(uint8_t &data)
{
    uint8_t local_data;
    
    int32_t rtn_val = readRegister(WHO_AM_I, local_data);
    if(rtn_val == 0)
    {
        data = local_data;
    }
    
    return rtn_val;
}


//******************************************************************************
int32_t H3LIS331DL::readControlRegisters(ControlRegisters_s &controlRegisters)
{
    uint8_t data[5];
    
    int32_t rtn_val = readBlock(CTRL_REG1, 5, data);
    if(rtn_val == 0)
    {
        uint8_t idx = 0;
        controlRegisters.reg1.all = data[idx++];
        controlRegisters.reg2.all = data[idx++];
        controlRegisters.reg3.all = data[idx++];
        controlRegisters.reg4.all = data[idx++];
        controlRegisters.reg5.all = data[idx++];
    }
    
    return rtn_val;
}


//******************************************************************************
int32_t H3LIS331DL::writeControlRegisters(
const ControlRegisters_s &controlRegisters)
{
    uint8_t data[5];
    uint8_t idx = 0;
    
    data[idx++] = controlRegisters.reg1.all;
    data[idx++] = controlRegisters.reg2.all;
    data[idx++] = controlRegisters.reg3.all;
    data[idx++] = controlRegisters.reg4.all;
    data[idx++] = controlRegisters.reg5.all;
    
    return writeBlock(CTRL_REG1, idx, data);
}


//******************************************************************************
int32_t H3LIS331DL::resetHPF()
{
    uint8_t data; //dummy data for read
    return readRegister(HP_FILTER_RESET, data);
}


//******************************************************************************
int32_t H3LIS331DL::readReferenceRegister(uint8_t &ref)
{
    uint8_t data;
    
    int32_t rtn_val = readRegister(REFERENCE, data);
    if(rtn_val == 0)
    {
        ref = data;
    }
    
    return rtn_val;
}


//******************************************************************************
int32_t H3LIS331DL::writeReferenceRegister(const uint8_t ref)
{
    return writeRegister(REFERENCE, ref);
}


//******************************************************************************
int32_t H3LIS331DL::readStatusRegister(STATUS_REG_u &status)
{
    uint8_t data;
    
    int32_t rtn_val = readRegister(STATUS_REG, data);
    if(rtn_val == 0)
    {
        status.all = data;
    }
    
    return rtn_val;
}


//******************************************************************************
int32_t H3LIS331DL::writeStatusRegister(const STATUS_REG_u &status)
{
    return writeRegister(STATUS_REG, status.all);
}


//******************************************************************************
int32_t H3LIS331DL::readAxis(Axis_s &axis)
{
    uint8_t data[6];
    
    int32_t rtn_val = readBlock(OUT_X_L, 6, data);
    if(rtn_val == 0)
    {
        uint8_t idx = 0;
        axis.x_axis = 0;
        axis.x_axis = data[idx++];
        axis.x_axis += (data[idx++] << 8);
        
        axis.y_axis = 0;
        axis.y_axis = data[idx++];
        axis.y_axis += (data[idx++] << 8);
        
        axis.z_axis = 0;
        axis.z_axis = data[idx++];
        axis.z_axis += (data[idx++] << 8);
    }
    
    return rtn_val;
}


//******************************************************************************
int32_t H3LIS331DL::readInterruptConfig(InterruptConfig_s &cfg, bool oneOrTwo)
{
    uint8_t data[4];
    Register_e reg = oneOrTwo ? INT1_CFG : INT2_CFG;
    
    int32_t rtn_val = readBlock(reg, 4, data);
    if(rtn_val == 0)
    {
        uint8_t idx = 0;
        cfg.cfg.all = data[idx++];
        cfg.src.all = data[idx++];
        cfg.threshold = data[idx++];
        cfg.duration = data[idx++];
    }
    
    return rtn_val;
}


//******************************************************************************
int32_t H3LIS331DL::writeInterruptConfig(const InterruptConfig_s &cfg, 
bool oneOrTwo)
{
    uint8_t data[4];
    Register_e reg = oneOrTwo ? INT1_CFG : INT2_CFG;
    uint8_t idx = 0;
    
    data[idx++] = cfg.cfg.all;
    data[idx++] = cfg.src.all;
    data[idx++] = cfg.threshold;
    data[idx++] = cfg.duration;
    
    return writeBlock(reg, idx, data);
}


//*************************** I2C Implementation ******************************
H3LIS331DL_I2C::H3LIS331DL_I2C(I2C &i2cBus, uint8_t slvAdrs):
m_i2cBus(i2cBus), m_w_adrs(slvAdrs << 1), m_r_adrs((slvAdrs << 1) | 1)
{
}


//*****************************************************************************
int32_t H3LIS331DL_I2C::readRegister(Register_e reg, uint8_t &data)
{
    char local_data[1];
    local_data[0] = reg;
    
    int32_t rtn_val = m_i2cBus.write(m_w_adrs, local_data, 1, true);
    if(rtn_val == 0)
    {
        rtn_val = m_i2cBus.read(m_r_adrs, local_data, 1);
        if(rtn_val == 0)
        {
            data = local_data[0];
        }
    }
    
    return rtn_val;
}


//*****************************************************************************
int32_t H3LIS331DL_I2C::writeRegister(Register_e reg, const uint8_t data)
{
    char local_data[2];
    local_data[0] = reg;
    local_data[1] = data;
    
    return m_i2cBus.write(m_w_adrs, local_data, 2);
}


//*****************************************************************************
int32_t H3LIS331DL_I2C::readBlock(Register_e startReg, 
const uint8_t readLength, uint8_t *data)
{
    char local_data[readLength];
    local_data[0] = startReg;
    
    int32_t rtn_val = m_i2cBus.write(m_w_adrs, local_data, 1, true);
    if(rtn_val == 0)
    {
        rtn_val = m_i2cBus.read(m_r_adrs, local_data, readLength);
        if(rtn_val == 0)
        {
            for(uint8_t idx = 0; idx < readLength; idx++)
            {
                data[idx] = local_data[idx];
            }
        }
    }
    
    return rtn_val;
}


//*****************************************************************************
int32_t H3LIS331DL_I2C::writeBlock(Register_e startReg, 
const uint8_t writeLength, const uint8_t *data)
{
    char local_data[writeLength + 1];
    local_data[0] = startReg;
    
    uint8_t idx = 1;
    for(; idx < (writeLength + 1); idx++)
    {
        local_data[idx] = data[idx];
    }
    
    return m_i2cBus.write(m_w_adrs, local_data, idx);
}


//*************************** SPI Implementation ******************************
H3LIS331DL_SPI::H3LIS331DL_SPI(SPI &spiBus, PinName cs):
m_spiBus(spiBus), m_cs(cs, 1)
{
}


//*****************************************************************************
int32_t H3LIS331DL_SPI::readRegister(Register_e reg, uint8_t &data)
{
    m_cs = 0;
    m_spiBus.write(R_BIT | reg);
    data = m_spiBus.write(0xFF);
    m_cs = 1;
    
    return 0;
}


//*****************************************************************************
int32_t H3LIS331DL_SPI::writeRegister(Register_e reg, const uint8_t data)
{
    m_cs = 0;
    m_spiBus.write(reg);
    m_spiBus.write(data);
    m_cs = 1;
    
    return 0;
}


//*****************************************************************************
int32_t H3LIS331DL_SPI::readBlock(Register_e startReg, 
const uint8_t readLength, uint8_t *data)
{
    m_cs = 0;
    m_spiBus.write(R_BIT | AUTO_INC_BIT | startReg);
    for(uint8_t idx = 0; idx < readLength; idx++)
    {
        data[idx] = m_spiBus.write(0xFF);
    }
    m_cs = 1;
    
    return 0;
}


//*****************************************************************************
int32_t H3LIS331DL_SPI::writeBlock(Register_e startReg, 
const uint8_t writeLength, const uint8_t *data)
{
    m_cs = 0;
    m_spiBus.write(AUTO_INC_BIT | startReg);
    for(uint8_t idx = 0; idx < writeLength; idx++)
    {
        m_spiBus.write(data[idx]);
    }
    m_cs = 1;
    
    return 0;
}

