//
// uVGAIII is a class to drive 4D Systems TFT touch screens
//
// Copyright (C) <2010> Stephane ROCHON <stephane.rochon at free.fr>
//
// uVGAIII is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// uVGAIII is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with uVGAIII.  If not, see <http://www.gnu.org/licenses/>.

// @author Stephane Rochon

#include "mbed.h"

// Debug Verbose on terminal enabled
#ifndef DEBUGMODE
#define DEBUGMODE 1
#endif

// Common WAIT value in millisecond
#define TEMPO 5

// 4DGL Functions values
// Graphic Commands values
#define CLS          0xFFCD
#define BCKGDCOLOR   0xFFA4
#define SCREENMODE   0xFF9E
#define CIRCLE       0xFFC3
#define CIRCLE_F     0xFFC2
#define TRIANGLE     0xFFBF
#define TRIANGLE_F   0xFFA9
#define LINE         0xFFC8
#define RECTANGLE    0xFFC5
#define RECTANGLE_F  0xFFC4
#define ELLIPSE      0xFFB2
#define ELLIPSE_F    0xFFB1
#define PUTPIXEL     0xFFC1
#define READPIXEL    0xFFC0
#define SCREENCOPY   0xFFAD
#define CLIPPING     0xFFA2
#define SETCLIPWIN   0xFFB5
#define EXTCLIPREG   0xFFB3
#define BUTTON       0x0011
#define PANEL        0xFFAF
#define SLIDER       0xFFAE
#define CHANGECOLOR  0xFFB4
#define MOVEORIGIN   0xFFCC
#define LINEPATTERN  0xFF9B
#define OUTLINECOLOR 0xFF9D
#define TRANSPARENCY 0xFFA0
#define TRANSPCOLOR  0xFFA1
#define PARAMETERS   0xFFCE

// System Commands
#define BAUDRATE     0x0026
#define SPEVERSION   0x001B
#define PMMCVERSION  0x001C
#define SETVOLUME    0xFF00

// Text and String Commands
#define MOVECURSOR   0xFFE9
#define PUTCHAR      0xFFFE
#define SETFONT      0xFFE5
#define TEXTOPACITY  0xFFDF
#define TEXTFGCOLOR  0xFFE7
#define TEXTBGCOLOR  0xFFE6
#define TEXTWIDTH    0xFFE4
#define TEXTHEIGHT   0xFFE3
#define TEXTXGAP     0xFFE2
#define TEXTYGAP     0xFFE1
#define TEXTBOLD     0xFFDE
#define TEXTINVERSE  0xFFDC
#define TEXTITALIC   0xFFDD
#define TEXTUNDLINE  0xFFDB
#define TEXTATTRIBU  0xFFDA
#define PUTSTRING    0x0018
#define CHARWIDTH    0x001E
#define CHARHEIGHT   0x001D
#define GETTOUCH     '\x6F'
#define WAITTOUCH    '\x77'
#define SETTOUCH     '\x75'

// Touch Screen Commands
#define TOUCHSET     0xFF38
#define TOUCHGET     0xFF37
#define TOUCHDETECT  0xFF39

// Slider
#define INTENDED     '\x00'
#define S_RAISED     '\x01'
#define HIDDEN       '\x02'

// Button
#define DEPRESSED    '\x00'
#define B_RAISED     '\x01'

// Panel
#define RECESSED     '\x00'
#define P_RAISED     '\x01'

// Screen answers
#define ACK          '\x06'
#define NAK          '\x15'

// Screen states
#define OFF          '\x00'
#define ON           '\x01'

// Graphics parameters
#define OBJECTCOLOR    18
#define RESOLUTION     32
#define PGDISPLAY      33
#define PGREAD         34
#define PGWRITE        35

// Graphics modes
#define SOLID        '\x00'
#define WIREFRAME    '\x01'

// Text modes
#define TRANSPARENT  '\x00'
#define OPAQUE       '\x01'

// Fonts Sizes
#define FONT1        '\x00'      // 7X8
#define FONT2        '\x01'      // 8X8
#define FONT3        '\x02'      // 8X12

// Touch Values
#define MOVE         '\x03'
#define STATUS       '\x00'
#define GETXPOSITION '\x01'
#define GETYPOSITION '\x02'

// Data speed
#define BAUD_110     0x0000
#define BAUD_300     0x0001
#define BAUD_600     0x0002
#define BAUD_1200    0x0003
#define BAUD_2400    0x0004
#define BAUD_4800    0x0005
#define BAUD_9600    0x0006
#define BAUD_14400   0x0007
#define BAUD_19200   0x0008
#define BAUD_31250   0x0009
#define BAUD_38400   0x000A
#define BAUD_56000   0x000B
#define BAUD_57600   0x000C
#define BAUD_115200  0x000D
#define BAUD_128000  0x000E
#define BAUD_256000  0x000F

// Defined Colors
#define BLACK   0x000000
#define WHITE   0xFFFFFF
#define LGREY   0xBFBFBF
#define DGREY   0x5F5F5F
#define RED     0xFF0000
#define LIME    0x00FF00
#define BLUE    0x0000FF
#define YELLOW  0xFFFF00
#define CYAN    0x00FFFF
#define MAGENTA 0xFF00FF
#define SILVER  0xC0C0C0
#define GRAY    0x808080
#define MAROON  0x800000
#define OLIVE   0x808000
#define GREEN   0x008000
#define PURPLE  0x800080
#define TEAL    0x008080
#define NAVY    0x000080

// Mode data
#define BACKLIGHT    '\x00'
#define DISPLAY      '\x01'
#define CONTRAST     '\x02'
#define POWER        '\x03'
#define ORIENTATION  '\x04'
#define TOUCH_CTRL   '\x05'
#define IMAGE_FORMAT '\x06'
#define PROTECT_FAT  '\x08'

// change this to your specific screen (newer versions) if needed
// Startup orientation is PORTRAIT so SIZE_X must be lesser than SIZE_Y
#define SIZE_X       480
#define SIZE_Y       800

#define IS_LANDSCAPE 0
#define IS_PORTRAIT  1

// Screen orientation
#define LANDSCAPE    '\x00'
#define LANDSCAPE_R  '\x01'
#define PORTRAIT     '\x02'
#define PORTRAIT_R   '\x03'

/**
 * This is a class for uVGAIII
 */
class uVGAIII : public Stream
{

public :

    uVGAIII(PinName tx, PinName rx, PinName rst);

// General Commands *******************************************************************************

/** Clear the entire screen using the current background colour */
    void cls();

/** Reset screen */
    void reset();
    
/** Set serial Baud rate (both sides : screen and mbed)
* @param Speed Correct BAUD value (see uVGAIII.h)
*/   
    void baudrate(int speed);

/** Set internal speaker to specified value
* @param value Correct range is 8 - 127
*/
    void set_volume(char value);

// Graphics Commands
/** Set the graphics orientation
* @param mode LANDSCAPE, LANDSCAPE_R, PORTRAIT, PORTRAIT_R
*/
    void screen_mode(char mode);
    
/**
*  Set background color of the screen
* @param color 24 bits in total with the most significant byte representing the red part and the middle byte representing the green part and the least significant byte representing the blue part
*/
    void background_color(int color);
/**
*  Draw a circle
* @param x x-coordinate of the center
* @param y y-coordinate of the center
* @param radius radius of the circle
* @param color color of the outline of the circle
*/
    void circle(int x , int y , int radius, int color);
/**
*  Draw a filled circle
* @param x x-coordinate of the center
* @param y y-coordinate of the center
* @param radius radius of the circle
* @param color color of the circle
*/
    void filled_circle(int x, int y, int radius, int color);
/**
*  Draw a triangle
* @param x1 x-coordinate of the first vertice
* @param y1 y-coordinate of the first vertice
* @param x2 x-coordinate of the second vertice
* @param y2 y-coordinate of the second vertice
* @param x3 x-coordinate of the third vertice
* @param y3 y-coordinate of the third vertice
* @param color color of the outline of the triangle
*/
    void triangle(int x1, int y1, int x2, int y2, int x3, int y3, int color);
/**
*  Draw a filled triangle
* @param x1 x-coordinate of the first vertice
* @param y1 y-coordinate of the first vertice
* @param x2 x-coordinate of the second vertice
* @param y2 y-coordinate of the second vertice
* @param x3 x-coordinate of the third vertice
* @param y3 y-coordinate of the third vertice
* @param color color of the triangle
*/
    void filled_triangle(int x1, int y1, int x2, int y2, int x3, int y3, int color);
/**
*  Draw a line
* @param x1 x-coordinate of the first vertice
* @param y1 y-coordinate of the first vertice
* @param x2 x-coordinate of the second vertice
* @param y2 y-coordinate of the second vertice
* @param color color of the line
*/
    void line(int x1, int y1 , int x2, int y2, int color);
/**
*  Draw a rectangle
* @param x1 x-coordinate of the first vertice
* @param y1 y-coordinate of the first vertice
* @param x2 x-coordinate of the second vertice
* @param y2 y-coordinate of the second vertice
* @param color color of the outline of the rectangle
*/
    void rectangle(int x1, int y1 , int x2, int y2, int color);
/**
*  Draw a filled rectangle
* @param x1 x-coordinate of the first vertice
* @param y1 y-coordinate of the first vertice
* @param x2 x-coordinate of the second vertice
* @param y2 y-coordinate of the second vertice
* @param color color of the rectangle
*/
    void filled_rectangle(int, int, int, int, int);
/**
*  Draw a ellipse
* @param x x-coordinate of the center
* @param y y-coordinate of the center
* @param radius_x x-radius of the ellipse
* @param redius_y y-radius of the ellipse
* @param color color of the outline of the rectangle
*/
    void ellipse(int x, int y , int radius_x, int radius_y, int color);
/**
*  Draw a filled ellipse
* @param x x-coordinate of the center
* @param y y-coordinate of the center
* @param radius_x x-radius of the ellipse
* @param redius_y y-radius of the ellipse
* @param color color of the rectangle
*/
    void filled_ellipse(int x, int y , int radius_x, int radius_y, int color);
/**
*  Draw a button
* @param state appearance of button, 0 = Button depressed; 1 = Button raised
* @param x x-coordinate of the top left corner of the button
* @param y y-coordinate of the top left corner of the button
* @param buttoncolor color of the button
* @param txtcolor color of the text
* @param font Specifies the Font ID
* @param txtWidth Set the width of the text
* @param txtHeight Set the height of the text
* @param text Specifies the text string
*/
    void button(int state, int x, int y, int buttoncolor, int txtcolor, int font, int txtWidth, int txtHeight, char * text);
/**
*  Draw a panel
* @param state appearance of panel, 0 = recessed; 1 = raised
* @param x x-coordinate of the top left corner of the button
* @param y y-coordinate of the top left corner of the button
* @param Width Set the width of the panel
* @param Height Set the height of the panel
* @param color Set the color of the panel
*/
    void panel(int state, int x, int y, int Width, int Height, int color);
/**
*  Draw a slider
* @param mode Mode = 0: Slider indented, mode = 1: Slider raised, mode 2: Slider hidden
* @param x1 x-coordinate of the top left position of the slider
* @param y1 y-coordinate of the top left position of the slider
* @param x2 x-coordinate of the bottom right position of the slider
* @param y2 y-coordinate of the bottom right position of the slider
* @param color Set the color of slider
* @param scale Set the full scale range of the slider for the thumb from 0 to n
* @param value If value positive, sets the relative position of the thumb on the slider bar, else set thumb to ABS position of the negative number
*/
    void slider(char mode, int x1, int y1, int x2, int y2, int color, int scale, int value);
/**
*  Draw a pixel
* @param x x-coordinate of the pixel
* @param y y-coordinate of the pixel
* @param color Set the color of pixel
*/
    void put_pixel(int x, int y, int color);
/**
*  Read the color value of the pixel
* @param x x-coordinate of the pixel
* @param y y-coordinate of the pixel
* @return the color value of the pixel
*/
    int  read_pixel(int x, int y);
/**
*  Copy an area of a screen
* @param xs x-coordinate of the top left corner of the area to be copied 
* @param ys y-coordinate of the top left corner of the area to be copied 
* @param xd x-coordinate of the top left corner of the area to be pasted 
* @param yd y-coordinate of the top left corner of the area to be pasted
* @param width Set the width of the copied area
* @param width Set the height of the copied area
*/
    void screen_copy(int xs, int ys , int xd, int yd , int width, int height);
/**
*  Enable or disable the ability for Clipping to be used
* @param value 0 = Clipping Disabled, 1 = Clipping Enabled
*/
    void clipping(char value);
/**
*  Specifiy the clipping window region on the screen
* @param x1 x-coordinate of the top left corner of the clipping window
* @param y1 y-coordinate of the top left corner of the clipping window
* @param x2 x-coordinate of the bottom right corner of the clipping window
* @param y2 y-coordinate of the bottom right corner of the clipping window
*/
    void set_clipping_win(int x1, int y1, int x2, int y2);
/**
*  Force the clip region to the extent of the last text
* 
*/
    void extend_clip_region();
/**
*  Change all old color pixels to new color within the clipping window area
* @param oldColor Sample color to be changed within the clipping window
* @param newColor New color to change all occurrences of old color within the clipping window
*/
    void change_color(int oldColor, int newColor);
/**
*  Move the origin to a new position
* @param xpos x-coordinate of the new origin
* @param ypos y-coordinate of the new origin
*/
    void move_origin(int xpos, int ypos);
/**
*  Set the line pattern
* @param pattern Correct value 0 to 65535 = number of bits in the line are turned off to form a pattern
*/
    void line_pattern(int pattern);
/**
*  Set the outline color for rectangles and circles
* @param color 24 bits in total of the color value
*/
    void outline_color(int color);
/**
*  Set whether to enable the transparency
* @param mode 0 = Transparency OFF, 1 = Transparency ON
*/
    void transparency(char mode);
/**
*  Set the color to be transparent
* @param color Color to be set to be transparent
*/
    void transparent_color(int color);
/**
*  Set graphics parameters
* @param function 18 = Object Color, 32 = Screen Resolution, 33 = Page Display, 34 = Page Read, 35 = Page Write
* @param value different values to be applied according to different functions
*/
    void graphics_parameters(int function, int value);

// Texts Commands
/**
*  Set font mode
* @param mode 0 for FONT1 = system font; 1 for FONT2; 2 for FONT3 = Default font
*/
    void set_font(char mode);
/**
*  Move teh text cursor to a screen postion
* @param line Line position
* @param column Column position
*/
    void move_cursor(int line, int column);
/**
*  Set whether or not the 'background' pixels are drawn
* @param mode 1 for ON(opaque), 0 for OFF(transparent)
*/
    void text_opacity(char mode);
/**
*  Set the text width multiplier between 1 and 16
* @param multiplier Width multiplier 1 to 16
*/
    void text_width(int multiplier);
/**
*  Set the text height multiplier between 1 and 16
* @param multiplier Height multiplier 1 to 16
*/
    void text_height(int multiplier);
/**
*  Set the pixel gap between characters(x-axis)
* @param pixelcount 0 to 32
*/
    void text_x_gap(int pixelcount);
/**
*  Set the pixel gap between characters(y-axis)
* @param pixelcount 0 to 32
*/
    void text_y_gap(int pixelcount);
/**
*  Set the bold attribute for the text
* @param mode 1 for ON, 0 for OFF
*/
    void text_bold(char mode);
/**
*  Inverse the background and foreground color of the text
* @param mode 1 for ON, 0 for OFF
*/
    void text_inverse(char mode);
/**
*  Set the text to italic
* @param mode 1 for ON, 0 for OFF
*/
    void text_italic(char mode);
/**
*  Set the text to underlined
* @param mode 1 for ON, 0 for OFF
*/
    void text_underline(char mode);
/**
*  Control several functions grouped, Text Bold, Text Italic, Text Inverse, Text Underlined
* @param value DEC 16 for BOLD, DEC 32 for ITALIC, DEC 64 for INVERSE, DEC 128 for UNDERLINED
*/
    void text_attributes(int value);
/**
*  Print a single character on the display
* @param c Character to be print on the screen
*/
    void put_char(char c);
/**
*  Print a string on the display
* @param s A Null terminated string
*/
    void put_string(char *s);
/**
*  Set the text foreground color
* @param color Specify the color to be set
*/
    void text_fgd_color(int color);
/**
*  Set the text background color
* @param color Specify the color to be set
*/
    void text_bgd_color(int color);
/**
*  calculate the width in pixel units for a character
* @param c Character for the width calculation
*/
    void char_width(char c);
/**
*  calculate the height in pixel units for a character
* @param c Character for the height calculation
*/
    void char_height(char c);
/** place char at current cursor position
*   used by virtual printf function _putc
* @param c Character to be printed out
*/
    void putc(char c);
/** place string at current cursor position
* @param s String to be printed out
*/
    void puts(char *s);

// Touch Command
/** Specify a new touch detect region on the screen
* @param x1 x-coordinate of the top left corner of the region
* @param y1 y-coordinate of the top left corner of the region
* @param x2 x-coordinate of the bottom right corner of the region
* @param y2 y-coordinate of the bottom rigth corner of the region
*/
    void detect_touch_region(int x1, int y1, int x2, int y2);
/** Get X coordinates of the touch
* @param x X coordinate of the touch
*/
    void touch_get_x(int *x);
/** Get Y coordinates of the touch
* @param y Y coordinate of the touch
*/
    void touch_get_y(int *y);
/** Set various touch screen related parameters
* @param mode Mode = 0: enable and initialise touch screen hardware; mode = 1: disable the touch screen; mode = 2: reset the current active region to default which is the full screen area
*/
    void touch_set(char mode);
/** Get the status of the screen
* @return Status of the screen: 0 = INVALID/NOTOUCH, 1 = PRESS, 2 = RELEASE, 3 = MOVING
*/
    int touch_status(void);

// Screen Version Data
    int version;
    int SPEversion;
    int PmmCversion;

// Text data
    char current_col;
    char current_row;
    int  current_color;
    char current_font;
    char current_orientation;
    char max_col;
    char max_row;
    int current_w, current_h;
    int current_fx, current_fy;

protected :

    Serial     _cmd;
    DigitalOut _rst;
    
    //used by printf
    virtual int _putc(int c) {
        putc(c);
        return 0;
    };
    virtual int _getc() {
        return -1;
    }

    void freeBUFFER  (void);
    void writeBYTE   (char);
    int  writeCOMMAND(char *, int, int);
    int  readVERSION (char *, int);
    void getTOUCH    (char *, int, int *);
    int  getSTATUS   (char *, int);
    void speversion  (void);
    void pmmcversion (void);
#if DEBUGMODE
    Serial pc;
#endif // DEBUGMODE
};

typedef unsigned char BYTE;

