//
// uVGAIII is a class to drive 4D Systems TFT touch screens
//
// Copyright (C) <2010> Stephane ROCHON <stephane.rochon at free.fr>
//
// uVGAIII is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// uVGAIII is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with uVGAIII.  If not, see <http://www.gnu.org/licenses/>.

#include "mbed.h"
#include "uVGAIII.h"

#define ARRAY_SIZE(X) sizeof(X)/sizeof(X[0])

//Serial pc(USBTX,USBRX);

//******************************************************************************************************
uVGAIII :: uVGAIII(PinName tx, PinName rx, PinName rst) : _cmd(tx, rx), 
                                                            _rst(rst) 
#if DEBUGMODE
                                                            ,pc(USBTX, USBRX)
#endif // DEBUGMODE
{ // Constructor

#if DEBUGMODE
    pc.baud(115200);

    pc.printf("\n\n\n");
    pc.printf("********************\n");
    pc.printf("uVGAIII CONSTRUCTOR\n");
    pc.printf("********************\n");
#endif
    _cmd.baud(9600);
    _rst = 1;    // put RESET pin to high to start TFT screen

    reset();
    cls();           // clear screen
    speversion();    // get SPE version information
    pmmcversion();   // get PmmC version information

    current_col         = 0;            // initial cursor col
    current_row         = 0;            // initial cursor row
    current_color       = WHITE;        // initial text color
    current_orientation = IS_LANDSCAPE;  // initial screen orientation

    set_font(FONT3);                 // initial font  ( This line can be removed because font has been set to be FONT3 )
    text_opacity(OPAQUE);            // initial text opacity  ( This line can be removed because text opacity has been set to be OPAQUE )
}

//******************************************************************************************************
void uVGAIII :: writeBYTE(char c) { // send a BYTE command to screen

    _cmd.putc(c);
    wait_ms(1);

#if DEBUGMODE
    pc.printf("   Char sent : 0x%02X\n",c);
#endif

}

//******************************************************************************************************
void uVGAIII :: freeBUFFER(void) {       // Clear serial buffer before writing command

    while (_cmd.readable()) _cmd.getc();  // clear buffer garbage
}

//******************************************************************************************************
int uVGAIII :: writeCOMMAND(char *command, int commNum, int respNum) { // send several BYTES making a command and return an answer

#if DEBUGMODE
    pc.printf("\n");
    pc.printf("New COMMAND : 0x%02X%02X\n", command[0], command[1]);
#endif
    int i, resp = 0;
    freeBUFFER();

    for (i = 0; i < commNum; i++) {writeBYTE(command[i]); wait_ms(100);} // send command to serial port

    for (i = 0; i < respNum; i++) {
       while (!_cmd.readable()) wait_ms(TEMPO);              // wait for screen answer
        if (_cmd.readable()) {
           resp = _cmd.getc();           // read response if any
#if DEBUGMODE
           pc.printf("response = 0x%02X\n",resp);
#endif
        }
    }
    return resp;
}

//**************************************************************************
void uVGAIII :: reset() {  // Reset Screen

    _rst = 0;               // put RESET pin to low
    wait_ms(TEMPO);         // wait a few milliseconds for command reception
    _rst = 1;               // put RESET back to high
    wait(3.2);                // wait 3s for screen to restart

    freeBUFFER();           // clean buffer from possible garbage
    
#if DEBUGMODE    
    pc.printf("Reset completed.\n");
#endif
}

//**************************************************************************
void uVGAIII :: speversion() {  // get SPE version
    char command[2] = "";
    int spe = 0;
    command[0] = (SPEVERSION >> 8) & 0xFF;
    command[1] = SPEVERSION & 0xFF;

    spe = readVERSION(command, 2);
    
    pc.printf("spe:%d\n",spe);
}

//**************************************************************************
void uVGAIII :: pmmcversion() {  // get PmmC version
    char command[2] = "";
    int pmmc = 0;
    command[0] = (PMMCVERSION >> 8) & 0xFF;
    command[1] = PMMCVERSION & 0xFF;

    pmmc = readVERSION(command, 2);
    
    pc.printf("pmmc:%d\n",pmmc);
}

//**************************************************************************
void uVGAIII :: baudrate(int speed) {  // set screen baud rate
    char command[4]= "";
    command[0] = ( BAUDRATE >> 8 ) & 0xFF;
    command[1] = BAUDRATE & 0xFF;
    switch (speed) {
        case  110 :
            command[2] = ( BAUD_110 >> 8 ) & 0xFF;
            command[3] = BAUD_110 & 0xFF;
            break;
        case  300 :
            command[2] = ( BAUD_300 >> 8 ) & 0xFF;
            command[3] = BAUD_300 & 0xFF;
            break;
        case  600 :
            command[2] = ( BAUD_600 >> 8 ) & 0xFF;
            command[3] = BAUD_600 & 0xFF;            
            break;
        case 1200 :
            command[2] = ( BAUD_1200>> 8 ) & 0xFF;
            command[3] = BAUD_1200 & 0xFF;
            break;
        case 2400 :
            command[2] = ( BAUD_2400 >> 8 ) & 0xFF;
            command[3] = BAUD_2400 & 0xFF;
            break;
        case 4800 :
            command[2] = ( BAUD_4800 >> 8 ) & 0xFF;
            command[3] = BAUD_4800 & 0xFF;
            break;
        case 9600 :
            command[2] = ( BAUD_9600 >> 8 ) & 0xFF;
            command[3] = BAUD_9600 & 0xFF;
            break;
        case 14400 :
            command[2] = ( BAUD_9600 >> 8 ) & 0xFF;
            command[3] = BAUD_9600 & 0xFF;
            break;
        case 19200 :
            command[2] = ( BAUD_19200 >> 8 ) & 0xFF;
            command[3] = BAUD_19200 & 0xFF;
            break;
        case 31250 :
            command[2] = ( BAUD_31250 >> 8 ) & 0xFF;
            command[3] = BAUD_31250 & 0xFF;
            break;
        case 38400 :
            command[2] = ( BAUD_38400 >> 8 ) & 0xFF;
            command[3] = BAUD_38400 & 0xFF;
            break;
        case 56000 :
            command[2] = ( BAUD_56000 >> 8 ) & 0xFF;
            command[3] = BAUD_56000 & 0xFF;
            break;
        case 57600 :
            command[2] = ( BAUD_57600 >> 8 ) & 0xFF;
            command[3] = BAUD_57600 & 0xFF;
            break;
        case 115200 :
            command[2] = ( BAUD_115200 >> 8 ) & 0xFF;
            command[3] = BAUD_115200 & 0xFF;
            break;
        case 128000 :
            command[2] = ( BAUD_128000 >> 8 ) & 0xFF;
            command[3] = BAUD_128000 & 0xFF;
            break;
        case 256000 :
            command[2] = ( BAUD_256000 >> 8 ) & 0xFF;
            command[3] = BAUD_256000 & 0xFF;
            break;
        default   :
            command[2] = ( BAUD_9600 >> 8 ) & 0xFF;
            command[3] = BAUD_9600 & 0xFF;
            speed = 9600;
            break;
    }
#if DEBUGMODE
    pc.printf("\n");
    pc.printf("New COMMAND : 0x%02X%02X\n", command[0], command[1]);
#endif
    int i, resp = 0;
    freeBUFFER();

    for (i = 0; i < 4; i++) writeBYTE(command[i]);      // send command to serial port
    wait_ms(5);
    _cmd.baud(speed);                                  // set mbed to same speed

    while (!_cmd.readable()) wait_ms(TEMPO);           // wait for screen answer
    if (_cmd.readable()) resp = _cmd.getc();           // read response if any

#if DEBUGMODE
    pc.printf("response = 0x%02X\n",resp);
#endif

#if DEBUGMODE
    pc.printf("Set baudrate completed.\n");
#endif
}

//******************************************************************************************************
int uVGAIII :: readVERSION(char *command, int number) { // read screen info and populate data

    int i, temp = 0, resp = 0;
    char response[5] = "";

#if DEBUGMODE
    pc.printf("\n");
    pc.printf("New COMMAND : 0x%02X%02X\n", command[0], command[1]);
#endif

    freeBUFFER();

    for (i = 0; i < number; i++) writeBYTE(command[i]);    // send all chars to serial port

    while (!_cmd.readable()) wait_ms(TEMPO);               // wait for screen answer

    while (_cmd.readable() && resp < ARRAY_SIZE(response)) {
        temp = _cmd.getc();
        response[resp] = (char)temp;
        pc.printf("response = 0x%02X\n",response[resp]);
        resp++;
    }
    switch (resp) {
        case 3 :                                           // if OK populate data and return version
            version  = int(response[1]) << 8 | response[2];
            break;
        default :  
            version = 0;                     // return 0                                  
            break;
    }
    return version;
}

//****************************************************************************************************
void uVGAIII :: set_volume(char value) {   // set sound volume to value
    char command[4] = "";

    command[0] = (SETVOLUME >> 8) & 0xFF;
    command[1] = SETVOLUME & 0xFF;
    
    command[2] = 0;
    command[3] = value;

    writeCOMMAND(command, 4, 1);
    
#if DEBUGMODE
    pc.printf("Sound volume completed.\n");
#endif
}

//******************************************************************************************************
void uVGAIII :: getTOUCH(char *command, int number, int *xy) { // read screen info and populate data

#if DEBUGMODE
    pc.printf("\n");
    pc.printf("New COMMAND : 0x%02X%02X\n", command[0], command[1]);
#endif
    int i, temp = 0, resp = 0;
    char response[5] = "";

    freeBUFFER();

    for (i = 0; i < number; i++) writeBYTE(command[i]);    // send all chars to serial port

    while (!_cmd.readable()) wait_ms(TEMPO);               // wait for screen answer

    while (_cmd.readable() && resp < ARRAY_SIZE(response)) {
        temp = _cmd.getc();
        response[resp++] = (char)temp;
    }

#if DEBUGMODE
    pc.printf("   Answer received %d : 0x%02X 0x%02X 0x%02X\n", resp, response[0], response[1], response[2]);
#endif

    switch (resp) {
        case 3 :                                                              // if OK populate data
            *xy = ((response[1]<<8)+ response[2]) * (response[1] != 0xFF);
            break;
        default :
            *xy = -1;
            break;
    }

#if DEBUGMODE
    pc.printf("   X or Y : %03d\n", *xy);
#endif
}

//******************************************************************************************************
int uVGAIII :: getSTATUS(char *command, int number) { // read screen info and populate data

#if DEBUGMODE
    pc.printf("\n");
    pc.printf("New COMMAND : 0x%02X%02X\n", command[0], command[1]);
#endif

    int i, temp = 0, resp = 0;
    char response[5] = "";

    freeBUFFER();

    for (i = 0; i < number; i++) writeBYTE(command[i]);    // send all chars to serial port

    while (!_cmd.readable()) wait_ms(TEMPO);    // wait for screen answer

    while (_cmd.readable() && resp < ARRAY_SIZE(response)) {
        temp = _cmd.getc();
        response[resp++] = (char)temp;
    }
    switch (resp) {
        case 3 :
            resp = (int)response[2];         // if OK populate data
            break;
        default :
            resp =  -1;                      // else return   0
            break;
    }
    
#if DEBUGMODE
    pc.printf("   Answer received : %d\n", resp);
#endif

    return resp;
}