#ifndef QUADCOPTER_H
#define QUADCOPTER_H

#include "mbed.h"
#include "Adafruit_9DOF.h"
#include "Serial_base.h"
#include "MRF24J40.h"
#include "Kalman.h"
#include "rtos.h"

#define FILTER_SIZE 1

#define MIN_PWM_1   0.1172
#define MIN_PWM_2   0.1161
#define MIN_PWM_3   0.1161
#define MIN_PWM_4   0.1171
#define OFF_PWM     0.1
#define MAX_PWM     0.15

// a few struct declarations. Can possibly eliminated using the vector class in STL
struct state {
    double phi; // roll
    double theta; // pitch
    double psi; // yaw
    double p; // omega x
    double q; // omega y
    double r; // omega z
};

struct filter {
    double phi[FILTER_SIZE]; // roll
    double theta[FILTER_SIZE]; // pitch
    double psi[FILTER_SIZE]; // yaw
    double p[FILTER_SIZE]; // omega x
    double q[FILTER_SIZE]; // omega y
    double r[FILTER_SIZE]; // omega z
};

struct controlInput {
    double f;
    double mx;
    double my;
    double mz;
};

typedef struct {
    double gyro_x;
    double gyro_y;
    double gyro_z;
    double roll;
    double pitch;
    double heading;
} offset;

struct motors {
    double m1;
    double m2;
    double m3;
    double m4;
};


class Quadcopter
{
private:
    Serial *pc_; // use for printing

    // allow for remote communication
    MRF24J40 *mrf_;
    int rcIterations_;
    int rcLength_;
    Timer rcTimer_;
    Timer *controlTimer;
    
    Mutex *desired_mutex;
    
    Kalman kalmanRoll;
    Kalman kalmanPitch;

    filter filters_;
    state state_;
    state desiredState_;
    double F_des_; // desired thrust force (excluding weight compensation)
    controlInput controlInput_;
    motors motorPwm_;

    // declarations for IMU use
    Adafruit_9DOF dof_;
    Adafruit_LSM303_Accel_Unified accel_;
    Adafruit_LSM303_Mag_Unified mag_;
    Adafruit_L3GD20_Unified gyro_;
    sensors_event_t accel_event_;
    sensors_event_t mag_event_;
    sensors_event_t gyro_event_;
    sensors_vec_t   orientation_;
    offset* initial_offsets_;

    double g_;  // gravity [m/s^2]
    double m_; // mass [kg]
    double l_; // arm lenght [m]
    double gamma_; // relation between moment coefficient and force coefficient.

    // proportional attitude control gains
    double kp_f_; // gain converting radio input to thrust.
    double kp_phi_;
    double kp_theta_;
    double kp_psi_;

    // derivative attitude control gains
    double kd_phi_;
    double kd_theta_;
    double kd_psi_;
    
    double ki_phi_;
    double ki_theta_;
    //double ki_psi_;
    double i_e_phi_;
    double i_e_theta_;
    
    // previous times
    float prev_controller_time_; 
    float prev_sensor_time;
    float start_sampling_time;
    
    double compAngleX;
    double compAngleY;
    double comp_x_offset;
    double comp_y_offset;

    double max_integral_phi_;
    double max_integral_theta_;


public:
    Quadcopter(Serial *pcPntr, MRF24J40 *mrfPntr, Timer* timer);  // constructor

    void controller();
    void readSensorValues();
    void readRc();

    motors getPwm();
    state getState();
    Adafruit_LSM303_Accel_Unified getAccel();
    //Adafruit_LSM303_Mag_Unified getMag();
    Adafruit_L3GD20_Unified getGyro();
    Adafruit_9DOF getIMU();
    offset* getOffset();
    double getForce();
};

#endif